<?php


  //////////////////////////////////////////////////////////////
  ///  @class Pair
  ///  @brief A simple container for a listname and an action
	///         (eg subscribe, unsubscribe) to perform on a list
  //////////////////////////////////////////////////////////////

	class Pair {
		var $action;   ///< Action to perform - s for subscribe, or u for unsubscribe.
		var $listname; ///< Name of the list to un/subscribe
	}



  //////////////////////////////////////////////////////////////
  ///
  ///  @class Action
  ///  @brief WebScriber Action class
  ///  @note  Copyright (c) 2005-2008 namesuppressed.
  ///
  /// -----------------------------------------------------------
  ///
  ///  Describes a list of actions to be performed, ie which
  ///  lists are to be subscribed to and which lists should
  ///  be unsubscribed from
  ///
  ///  The Action represents a set of tasks that a Visitor would
  ///  like to take on their mailing list subscriptions.  The
  ///  visitor might like to subscribe and unsubscribe from a
  ///  number of lists.  All of these subscriptions can be part
  ///  of a single Action.  Only one Action can be performed in
  ///  a Transaction.
  ///
  ///  There is no need to call the Action directly.  The
  ///  Transaction class is the only class that needs to call
  ///  the Action.
  ///
  ///  Why is the Action a separate class to the Transaction?
  ///  Because it makes coding the Transaction class a little
  ///  bit easier.  Perhaps the Action and Transaction classes
  ///  can be amalgamated one day.
  ///
  /// -----------------------------------------------------------
  ///
  ///  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
  ///  KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
  ///  WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
  ///  PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
  ///  OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
  ///  OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
  ///  OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
  ///  SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
  ///
  //////////////////////////////////////////////////////////////



class Action {

  var $listoflists;  ///< Array of un/subscribe tasks to perform



	//////////////////////////////////////////////////////////////
	///  Constructor for the Action class
	///
	///  @return an Action object
	//////////////////////////////////////////////////////////////

  function Action() {
    $this->listoflists = array();
  }



  //////////////////////////////////////////////////////////////
  ///  Add a list to subscribe to this Action
  ///
  ///  @param name  the name of the list to subscribe to
  //////////////////////////////////////////////////////////////
  
  function subscribeList($name) {
    $inarray = false;
    $name = strtolower($name);
    for ($i = 0; $i < count($this->listoflists); $i++) {
      if ($this->listoflists[$i]->listname == $name) {
        $inarray = true;
        if ($this->listoflists[$i]->action == 'u')
           $this->listoflists[$i]->action = 's';
      }
    }
    if (!$inarray) {
      $temp = new Pair();
      $temp->action = 's';
      $temp->listname = $name;
      array_push($this->listoflists, $temp);
    }
  }



  //////////////////////////////////////////////////////////////
  ///  Add a list to unsubscribe from to this Action
  ///
  ///  @param name  the name of the list to unsubscribe from
  //////////////////////////////////////////////////////////////
  
  function unsubscribeList($name) {
    $inarray = false;
    $name = strtolower($name);
    for ($i = 0; $i < count($this->listoflists); $i++) {
      if ($this->listoflists[$i]->listname == $name) {
        $inarray = true;
        if ($this->listoflists[$i]->action == 's')
           $this->listoflists[$i]->action = 'u';
      }
    }
    if (!$inarray) {
      $temp = new Pair();
      $temp->action = 'u';
      $temp->listname = $name;
      array_push($this->listoflists, $temp);
    }
  }



  //////////////////////////////////////////////////////////////
  ///  Get the number of subscribe & unsubscribe tasks
  ///  associated with this Action
  ///
  ///  @return the number of tasks associated with this Action
  //////////////////////////////////////////////////////////////
  
  function getNumTasks() {
    return count($this->listoflists);
  }



  //////////////////////////////////////////////////////////////
  ///  Get a particular task from the list of tasks associated
  ///  with this Action
  ///
  ///  @param  val  the number of the task to get
  ///  @return the task requested, or false if val is out of
  ///          range
  //////////////////////////////////////////////////////////////
  
  function getTask($val) {
    if ($val <= count($this->listoflists) && $val >= 0)
      return $this->listoflists[$val];
    else return false;
  }



  //////////////////////////////////////////////////////////////
  ///  Get all the subscribe tasks in this action
  ///
  ///  @return an array containing all subscribe tasks
  //////////////////////////////////////////////////////////////
  
  function getToSubscribe() {
    $temparray = array();
    for ($i = 0; $i < $this->getNumTasks(); $i++) {
      $listname = $this->listoflists[$i]->listname;
      if ($this->listoflists[$i]->action == 's')
        array_push($temparray, $listname);
    }
    return $temparray;
  }



  //////////////////////////////////////////////////////////////
  ///  Get all the unsubscribe tasks in this action
  ///
  ///  @return an array containing all unsubscribe tasks
  //////////////////////////////////////////////////////////////
  
  function getToUnsubscribe() {
    $temparray = array();
    for ($i = 0; $i < $this->getNumTasks(); $i++) {
      $listname = $this->listoflists[$i]->listname;
      if ($this->listoflists[$i]->action == 'u')
        array_push($temparray, $listname);
    }
    return $temparray;
  }



  //////////////////////////////////////////////////////////////
  ///  Set all the member variables of this object by parsing
  ///  a string representation of the object.
  ///
  ///  @see   toString
  ///  @param str  the string to parse for data
  //////////////////////////////////////////////////////////////

  function fromString($str) {
    $this->listoflists = array();
    $elements = explode(",", trim($str));
    for ($i = 0; $i < count($elements); $i++) {
      $fragments = explode(":", trim($elements[$i]));
      $temp = new Pair();
      $temp->action = $fragments[0];
      $temp->listname = $fragments[1];
      array_push($this->listoflists, $temp);
    }
  }



  //////////////////////////////////////////////////////////////
  ///  Returns a string containing this Action's info
  ///
  ///  @see    fromString
  ///  @return a string representation of this object
  //////////////////////////////////////////////////////////////

  function toString() {
    $retval = "";
    for ($i = 0; $i < count($this->listoflists); $i++) {
      if ($i > 0) $retval .= ",";
      $retval .= $this->listoflists[$i]->action;
      $retval .= ":";
      $retval .= $this->listoflists[$i]->listname;
    }
    return $retval;
  }
}
?>