<?php

  //////////////////////////////////////////////////////////////
  ///
  ///  @class Configuration
  ///  @brief WebScriber Configuration class
  ///  @note  Copyright (c) 2005-2008 namesuppressed.
  ///
  ///  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
  ///  KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
  ///  WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
  ///  PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
  ///  OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
  ///  OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
  ///  OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
  ///  SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
  ///
  //////////////////////////////////////////////////////////////


  require_once('maillist.php');


  class Configuration {


    var $lists;        ///< Array of lists


    var $days;         ///< number of days that a token is valid
    var $maxemails;    ///< maximum number of emails that can be sent
    var $path;         ///< path to ezmlm functions
    var $listbase;     ///< path where ezmlm directories are stored
    var $basedir;      ///< where configuration files are stored

    var $scripturl;    ///< url to this script page
    var $testingmode;  ///< set this to true to disable sending mail
    var $singleoptout; ///< set true to unsubscribe without confirmation

    // SMTP mailout related variables
    var $smtphost;     ///< The address/domain name of the SMTP host
    var $smtpusername; ///< The username we're sending email through
    var $smtppassword; ///< The password for the SMTP username
    var $smtprealm;    ///< The realm we use to send SMTP email
    var $smtpdebug;    ///< Output SMTP debugging messages?
    var $builtinmail;  ///< Use built-in mail functions instead of SMTP?
    var $emailcopies;  ///< if true, send copies of all emails to self

    // Useful handy variables for emails
    var $sendername;   ///< Name to put in From field when sending emails
    var $senderemail;  ///< From email address when sending emails

    // Subject lines for each email message
    var $batchSubject;          ///< Subject line for Batch Subscription emails
    var $subscribeSubject;      ///< Subject line for Subscribe emails
    var $unsubscribeSubject;    ///< Subject line for Unsubscribe emails
    var $subConfirmSubject;     ///< Subject line for Subscription Confirmation emails
    var $unsubConfirmSubject;   ///< Subject line for Unsubcription Confirmation emails
    var $updateSubject;         ///< Subject line for Update emails
    var $updateConfirmSubject;  ///< Subject line for Update Confirmation emails

    // Password & Unlocking code
    var $password;     ///< For triggering confirmation emails with notes
    var $unlockcode;   ///< A very flimsy implementation of an unlock code

    // Debugging
    var $debugmode;      ///< Boolean, should we enable debugging?
    var $configErrors;   ///< Array of text descriptions, errors in config

    // Banned emails and domains
    var $banneddomains;  ///< Array of banned domains
    var $bannedemails;   ///< Array of banned email addresses
    





    //////////////////////////////////////////////////////////////
    ///  Default constructor for the Configuration object class.
    ///
    ///  @return  A Configuration object that you can call to run
    ///           the functions contained within it.
    //////////////////////////////////////////////////////////////

    function Configuration() {

      $this->lists = array();

      $this->scripturl    = 'http://example.com/cgi-bin/lists/email.php';
      $this->path         = '/usr/local/bin/ezmlm/';
      $this->basedir      = '/big/dom/xexample/cgi-bin/lists/';
      $this->listbase     = '/big/dom/xexample/username/';

      $this->smtphost     = 'localhost';
      $this->smtpusername = '';
      $this->smtppassword = '';
      $this->smtprealm    = '';
      $this->smtpdebug    = false;
      $this->builtinmail  = false;

      $this->password     = 'default';
      $this->unlockcode   = '';

      $this->sendername   = 'Jane Q Public';
      $this->senderemail  = 'support@example.com';

      $this->batchSubject         = 'Invitation to join &&LISTNAME&&';
      $this->subscribeSubject     = 'Action required to subscribe to &&LISTNAME&&';
      $this->unsubscribeSubject   = 'Action required to unsubscribe from &&LISTNAME&&';
      $this->subConfirmSubject    = 'Thank you! Subscribed to &&LISTNAME&&';
      $this->updateSubject        = 'Action required to modify subscriptions';
      $this->unsubConfirmSubject  = 'Unsubscribed from &&LISTNAME&&';
      $this->updateConfirmSubject = 'Thank you! Updated subscriptions';

      $this->days         = 3;
      $this->maxemails    = 500;
      $this->emailcopies  = false;
      $this->testingmode  = false;
      $this->singleoptout = true;
      $this->debugmode    = false;

      $this->banneddomains = array();
      $this->bannedemails  = array();
      
      $this->configErrors  = array();
    }





    //////////////////////////////////////////////////////////////
    ///  Checks for any errors in the configuration.
    ///
    ///  @todo   This function has not been implemented.
    ///  @return false if a problem was detected in the config,
    ///          true otherwise.
    //////////////////////////////////////////////////////////////

    function testConfigurationOkay() {
      $this->configErrors = array();
      
      // Check the password is acceptable
      if ($this->passwordIsDefault())   array_push($this->configErrors, "You have not changed your password from the default value. You must do this for security reasons.  Please edit your config.txt file and modify the value of the PASSWORD variable.");
      if (!$this->passwordLengthOkay()) array_push($this->configErrors, "The password you have set is too short.  For security, passwords must be a minimum of 6 characters long.  Please edit your config.txt file and modify the value of the PASSWORD variable accordingly.");
      
      // Let's try to be extra helpful - check if the last character
      // of the basedirectory is a forward slash where necessary.
      if (!$this->stringEndsWithSlash($this->path))     array_push($this->configErrors, "It looks like you forgot the trailing slash at the end of the PATH_TO_EZMLM variable.");
      if (!$this->stringEndsWithSlash($this->basedir))  array_push($this->configErrors, "It looks like you forgot the trailing slash at the end of the BASEDIRECTORY variable.");
      if (!$this->stringEndsWithSlash($this->listbase)) array_push($this->configErrors, "It looks like you forgot the trailing slash at the end of the LIST_BASEDIR variable.");
      if ($this->stringEndsWithSpace($this->path))     array_push($this->configErrors, "It looks like there's an extra space at the end of the PATH_TO_EZMLM variable. Please delete it, and ensure it ends in a trailing slash.");
      if ($this->stringEndsWithSpace($this->basedir))  array_push($this->configErrors, "It looks like there's an extra space at the end of the BASEDIRECTORY variable. Please delete it, and ensure it ends in a trailing slash.");
      if ($this->stringEndsWithSpace($this->listbase)) array_push($this->configErrors, "It looks like there's an extra space at the end of the LIST_BASEDIR variable. Please delete it, and ensure it ends in a trailing slash.");

      // Check the email templates are okay
      if (!$this->testEmailTemplateFiles()) {
        $tmperror  = 'There was a problem loading the email template files required by this ';
        $tmperror .= 'script.  Please check that the BASEDIRECTORY variable in your config.txt ';
        $tmperror .= 'configuration file is correct, and that each of the email template files ';
        $tmperror .= 'exists and is readable.  You should find them in the email directory where ';
        $tmperror .= 'you installed the script.  If you are unsure what is wrong, the readme ';
        $tmperror .= 'file may be able to help you.';
        if (strpos(PHP_OS, "Darwin") !== false) {
          $tmperror .= '<br /><br /> Your server appears to be running ';
          $tmperror .= 'Mac OS X.  Try changing the Ownership & Permissions on your ';
          $tmperror .= 'email templates and the folder they are in - click on the file, ';
          $tmperror .= 'choose Get Info from the file ';
          $tmperror .= 'menu, and change the Group on each file to \'www\'.  Also';
          $tmperror .= 'change the Access permissions to Read &amp; Write.';
        }  
        $tmperror .= '<br><br><pre>';
        $tmperror .= 'batchsubscribe:  ' . $this->basedir . 'email/batchsubscribe.txt' . ' <br />';
        $tmperror .= 'subscribe:       ' . $this->basedir . 'email/subscribe.txt' . ' <br />';
        $tmperror .= 'unsubscribe:     ' . $this->basedir . 'email/unsubscribe.txt' . ' <br />';
        $tmperror .= 'subconfirmed:    ' . $this->basedir . 'email/subconfirmed.txt' . ' <br />';
        $tmperror .= 'unsubconfirmed:  ' . $this->basedir . 'email/unsubconfirmed.txt' . ' <br />';
        $tmperror .= 'updated:         ' . $this->basedir . 'email/update.txt' . ' <br />';
        $tmperror .= 'updateconfirmed: ' . $this->basedir . 'email/updateconfirmed.txt' . ' <br />';
        $tmperror .= '</pre>';
        array_push($this->configErrors, $tmperror);
      }

      return (count($this->configErrors) == 0);
    }



    //////////////////////////////////////////////////////////////
    ///  Returns an array of strings, describing config errors
    ///
    ///  @return an array of strings, hints to fix the config file
    //////////////////////////////////////////////////////////////

    function getConfigurationErrors() {
      return $this->configErrors;
    }



    //////////////////////////////////////////////////////////////
    ///  Tests if the template files can be read
    ///
    ///  @return  true if the logfiles are readable and writable,
    ///           false otherwise
    //////////////////////////////////////////////////////////////

    function testEmailTemplateFiles() {
      $emailtemplatesok  = is_readable($this->basedir . 'email/batchsubscribe.txt');
      $emailtemplatesok &= is_readable($this->basedir . 'email/subscribe.txt');
      $emailtemplatesok &= is_readable($this->basedir . 'email/unsubscribe.txt');
      $emailtemplatesok &= is_readable($this->basedir . 'email/subconfirmed.txt');
      $emailtemplatesok &= is_readable($this->basedir . 'email/unsubconfirmed.txt');
      $emailtemplatesok &= is_readable($this->basedir . 'email/update.txt');
      $emailtemplatesok &= is_readable($this->basedir . 'email/updateconfirmed.txt');
      return $emailtemplatesok;
    }



    //////////////////////////////////////////////////////////////
    ///  Checks if the password is an appropriate length.
    ///
    ///  @return false if password is too short, otherwise true
    //////////////////////////////////////////////////////////////

    function passwordLengthOkay() {
       return (strlen($this->password) >= 6);
    }



    //////////////////////////////////////////////////////////////
    ///  Checks if the password is still the default value
    ///
    ///  @return true if password is default, false otherwise
    //////////////////////////////////////////////////////////////

    function passwordIsDefault() {
       return ($this->password == 'default');
    }



    //////////////////////////////////////////////////////////////
    ///  Checks if the variable ends with a trailing slash
    ///
    ///  @param  the string to check for a trailing slash
    ///  @return true if string has trailing slash, else false
    //////////////////////////////////////////////////////////////

    function stringEndsWithSlash($str) {
      $lastchar = $str[strlen($str) - 1];
      return ($lastchar == '/' || $lastchar == '\\');
    }



    //////////////////////////////////////////////////////////////
    ///  Checks if the variable ends with a space
    ///
    ///  @param  the string to check for a trailing space
    ///  @return true if string has trailing space, else false
    //////////////////////////////////////////////////////////////

    function stringEndsWithSpace($str) {
      $lastchar = $str[strlen($str) - 1];
      return ($lastchar == ' ');
    }



    //////////////////////////////////////////////////////////////
    ///  Retrieves the value for a configuration setting
    ///
    ///  @param contents   array of strings to search for id
    ///  @param id         the id of the config setting to get
    ///  @param variable   reference to the variable to set
    ///  @return false if an error occurred, true otherwise.
    //////////////////////////////////////////////////////////////

    function getConfigSetting($contents, $id, &$variable) {
      $value = '';
      preg_match("/\n" . $id . "=(.*)\n/i", $contents, $temparray);
      if (count($temparray) > 0) {
        $temp = trim($temparray[1]);
        $value = $temp;
        if (strtolower($temp) == 'on')  $value = true;
        if (strtolower($temp) == 'off') $value = false;
      }
      else {
        // No value given, so set a default.  We should refactor
        // this code so each variable has an ID, default value
        // and a description.  Probably.
        if (is_bool($variable)) $value = false;
        if (is_numeric($variable)) $value = 0;
      }
      $variable = $value;
    }



    //////////////////////////////////////////////////////////////
    ///  Constructs a correctly formatted line in the
    ///  configuration file.  Includes support for correctly
    ///  writing out boolean values.
    ///
    ///  @param id        the id of the config setting to get
    ///  @param variable  reference to the variable to set
    ///  @return string representation of id and its value
    //////////////////////////////////////////////////////////////

    function constructConfigLine($id, &$variable) {
      $configline = $id . '=';
      if (is_bool($variable)) {
        if ($variable == true)       $configline .= 'on';
        else if ($variable == false) $configline .= 'off';
      }
      else $configline .= $variable;
      $configline .= "\n";
      return $configline;
    }



    //////////////////////////////////////////////////////////////
    ///  Loads the configuration data from a file.
    ///
    ///  @param  fn  the full path to the configuration file.
    ///  @return false if an error occurred, true otherwise.
    //////////////////////////////////////////////////////////////

    function load($fn) {

      trigger_error("Inside configuration->load.");

      if (!is_readable(realpath($fn))) return false;

      $fd = fopen ($fn, "rb");
      $contents = fread($fd, filesize($fn));
      fclose($fd);

      // Replace Windows & Mac line endings
      $contents = ereg_replace ("\r\n", "\n", $contents);
      $contents = ereg_replace ("\r", "\n", $contents);

      $this->getConfigSetting($contents, 'SCRIPT_URL',           $this->scripturl);
      $this->getConfigSetting($contents, 'PATH_TO_EZMLM',        $this->path);
      $this->getConfigSetting($contents, 'BASEDIRECTORY',        $this->basedir);
      $this->getConfigSetting($contents, 'LIST_BASEDIR',         $this->listbase);
      $this->getConfigSetting($contents, 'DAYS_VALID',           $this->days);
      $this->getConfigSetting($contents, 'MAX_EMAILS',           $this->maxemails);
      $this->getConfigSetting($contents, 'SCRIPT_URL',           $this->scripturl);
      $this->getConfigSetting($contents, 'TESTING_MODE',         $this->testingmode);
      $this->getConfigSetting($contents, 'SINGLE_OPT_OUT',       $this->singleoptout);
      $this->getConfigSetting($contents, 'SENDER_NAME',          $this->sendername);
      $this->getConfigSetting($contents, 'SENDER_EMAIL',         $this->senderemail);
      $this->getConfigSetting($contents, 'UNLOCKING_CODE',       $this->unlockcode);
      $this->getConfigSetting($contents, 'PASSWORD',             $this->password);
      $this->getConfigSetting($contents, 'EMAIL_COPIES_TO_SELF', $this->emailcopies);
      $this->getConfigSetting($contents, 'SMTP_HOST_NAME',       $this->smtphost);
      $this->getConfigSetting($contents, 'SMTP_USERNAME',        $this->smtpusername);
      $this->getConfigSetting($contents, 'SMTP_PASSWORD',        $this->smtppassword);
      $this->getConfigSetting($contents, 'SMTP_REALM',           $this->smtprealm);
      $this->getConfigSetting($contents, 'SMTP_DEBUG',           $this->smtpdebug);
      $this->getConfigSetting($contents, 'USE_BUILTIN_MAIL',     $this->builtinmail);
      $this->getConfigSetting($contents, 'DEBUG_MODE',           $this->debugmode);

      $this->getConfigSetting($contents, 'BATCH_SUBJECT',        $this->batchSubject);
      $this->getConfigSetting($contents, 'SUBSCRIBE_SUBJECT',    $this->subscribeSubject);
      $this->getConfigSetting($contents, 'UNSUBSCRIBE_SUBJECT',  $this->unsubscribeSubject);
      $this->getConfigSetting($contents, 'UPDATE_SUBJECT',       $this->updateSubject);
      $this->getConfigSetting($contents, 'SUBCONFIRM_SUBJECT',   $this->subConfirmSubject);
      $this->getConfigSetting($contents, 'UNSUBCONFIRM_SUBJECT', $this->unsubConfirmSubject);
      $this->getConfigSetting($contents, 'UPDATECONFIRM_SUBJECT',$this->updateConfirmSubject);

      $expression = "/(\[.*?\][ \t]*\nNAME=\".*\"[ \t]*\nDESCRIPTION=\".*\"[ \t]*\nPRIVATE=\".*\"[ \t]*\nSUBSCRIBE=\".*\"[ \t]*\nUNSUBSCRIBE=\".*\"[ \t]*)/";
      $expression2 = "/\[(.*?)\][ \t]*\nNAME=\"(.*)\"[ \t]*\nDESCRIPTION=\"(.*)\"[ \t]*\nPRIVATE=\"(.*)\"[ \t]*\nSUBSCRIBE=\"(.*)\"[ \t]*\nUNSUBSCRIBE=\"(.*)\"[ \t]*/";
      preg_match_all($expression, $contents, $out, PREG_PATTERN_ORDER);

      $tempdomains = '';
      $this->getConfigSetting($contents, 'BANNED_DOMAINS', $tempdomains);
      $this->banneddomains = explode(",", $tempdomains);
      $this->banneddomains = array_map("strtolower", $this->banneddomains);
      $this->banneddomains = array_map("trim",       $this->banneddomains);

      $tempemails = '';
      $this->getConfigSetting($contents, 'BANNED_EMAILS',  $tempemails);
      $this->bannedemails = explode(",", $tempemails);
      $this->bannedemails = array_map("strtolower", $this->bannedemails);
      $this->bannedemails = array_map("trim",       $this->bannedemails);

      $this->lists = array();
      $count = 0;
      foreach ($out[1] as $item) { $count++; }
      for($i = 0; $i < $count; $i++) {
        preg_match($expression2, $out[1][$i], $matches);
        if ($matches) {
          $listitem = new Maillist();
          $listitem->setID($matches[1]);
          $listitem->setName($matches[2]);
          $listitem->setDescription($matches[3]);
          if ($matches[4] == "true") $listitem->setPrivate(true);
          else $listitem->setPrivate(false);
          $listitem->setSubscribeEmail($matches[5]);
          $listitem->setUnsubscribeEmail($matches[6]);
          array_push($this->lists, $listitem);
        }
      }

      trigger_error("Leaving configuration->loadConfiguration.");
      return true;
    }



    //////////////////////////////////////////////////////////////
    ///  Saves the configuration data to a file
    ///
    ///  @param  fn  full path to save the configuration file.
    ///  @return false if an error occurred, true otherwise.
    //////////////////////////////////////////////////////////////

    function save($fn) {

      trigger_error("Inside configuration->save.");

      $config  = "###########################################################\n";
      $config .= "# WebScriber configuration file                           #\n";
      $config .= "# 2002-2008 namesuppressed                               #\n";
      $config .= "###########################################################\n\n\n";

      $config .= "# In your web browser, what address would you need to type\n";
      $config .= "# in to see the script?  This will depend on where you have\n";
      $config .= "# installed it on your webserver.  You should type this\n";
      $config .= "# script URL below.\n\n";
      $config .= $this->constructConfigLine('SCRIPT_URL',           $this->scripturl);
      $config .= "\n\n";

      $config .= "# This is the path to the ezMLM executable files.  They are\n";
      $config .= "# usually stored in /usr/local/bin/ezmlm/ or /usr/bin/ezmlm\n";
      $config .= "# If using ezMLM-IDX (eg if your site is at FutureQuest),\n";
      $config .= "# try /usr/local/ezmlm-idx/bin/ instead.\n\n";
      $config .= $this->constructConfigLine('PATH_TO_EZMLM',        $this->path);
      $config .= "\n\n";

      $config .= "# The Basedirectory is where you have stored all your\n";
      $config .= "# important files like HTML templates and log files, things\n";
      $config .= "# that you don't want hackers to touch and aren't accessed\n";
      $config .= "# directly by users. Leave a trailing slash on the end of\n";
      $config .= "# the directory.\n\n";
      $config .= $this->constructConfigLine('BASEDIRECTORY',        $this->basedir);
      $config .= "\n\n";

      $config .= "# This is where the ezMLM mailing list files are stored on\n";
      $config .= "# your webserver.  Make sure you add a slash at the end!\n\n";
      $config .= $this->constructConfigLine('LIST_BASEDIR',         $this->listbase);
      $config .= "\n\n";

      $config .= "# The SMTP_HOST_NAME is the name of the SMTP server you\n";
      $config .= "# send email through - this could be example.com,\n";
      $config .= "# smtp.example.com, mail.example.com or something else.\n\n";
      $config .= $this->constructConfigLine('SMTP_HOST_NAME',       $this->smtphost);
      $config .= "\n\n";

      $config .= "# The username and password are the ones you use in your\n";
      $config .= "# email program to send email.  The realm is probably\n";
      $config .= "# your domain name.  If requested you can turn SMTP_DEBUG\n";
      $config .= "# to 'on' to help track down email problems.  You should\n";
      $config .= "# also turn on DEBUG_MODE below to record the debug info\n";
      $config .= "# to a debug logfile.  Otherwise, depending how your web\n";
      $config .= "# server is configured, the errors may not be displayed.\n\n";
      $config .= $this->constructConfigLine('SMTP_USERNAME',        $this->smtpusername);
      $config .= $this->constructConfigLine('SMTP_PASSWORD',        $this->smtppassword);
      $config .= $this->constructConfigLine('SMTP_REALM',           $this->smtprealm);
      $config .= $this->constructConfigLine('SMTP_DEBUG',           $this->smtpdebug);
      $config .= "\n\n";

      $config .= "# If you're still having problems with the SMTP servers,\n";
      $config .= "# you can use PHP's built-in mail system as a last resort\n";
      $config .= "# by changing 'off' to 'on' below.  However, using the\n";
      $config .= "# PHP mail system could cause server problems when sending\n";
      $config .= "# lots of emails at once.\n\n";
      $config .= $this->constructConfigLine('USE_BUILTIN_MAIL',     $this->builtinmail);
      $config .= "\n\n";

      $config .= "# Sometimes you need to \"trigger\" a subscription manually\n";
      $config .= "# when a user agrees to join your mailing list by non-web\n";
      $config .= "# methods (eg by ticking a box on a paper form they mailed\n";
      $config .= "# to you).  This script allows you to trigger subscription\n";
      $config .= "# confirmations by using a password, which you MUST set\n";
      $config .= "# below.  (The default will not be accepted).\n\n";
      $config .= $this->constructConfigLine('PASSWORD',             $this->password);
      $config .= "\n\n";

      $config .= "# When you purchase the script you will be given an\n";
      $config .= "# unlocking code.  Type the code here and you won't get\n";
      $config .= "# the \"unregistered\" notice at the end of your emails.\n\n";
      $config .= $this->constructConfigLine('UNLOCKING_CODE',       $this->unlockcode);
      $config .= "\n\n";

      $config .= "# Users sometimes reply back to confirmation emails the\n";
      $config .= "# script sends out - it is good to have a real human reply\n";
      $config .= "# to these emails.  For this reason, you should list your\n";
      $config .= "# real name and a contact email address below.\n\n";
      $config .= $this->constructConfigLine('SENDER_NAME',          $this->sendername);
      $config .= $this->constructConfigLine('SENDER_EMAIL',         $this->senderemail);
      $config .= "\n\n";

      $config .= "# These are the subject lines for emails that are sent out\n";
      $config .= "# to your subscribers.  You can modify them, and you'll\n";
      $config .= "# find a list of variables you can use in the readme file.\n\n";
      $config .= $this->constructConfigLine('BATCH_SUBJECT',        $this->batchSubject);
      $config .= $this->constructConfigLine('SUBSCRIBE_SUBJECT',    $this->subscribeSubject);
      $config .= $this->constructConfigLine('UNSUBSCRIBE_SUBJECT',  $this->unsubscribeSubject);
      $config .= $this->constructConfigLine('UPDATE_SUBJECT',       $this->updateSubject);
      $config .= $this->constructConfigLine('SUBCONFIRM_SUBJECT',   $this->subConfirmSubject);
      $config .= $this->constructConfigLine('UNSUBCONFIRM_SUBJECT', $this->unsubConfirmSubject);
      $config .= $this->constructConfigLine('UPDATECONFIRM_SUBJECT',$this->updateConfirmSubject);
      $config .= "\n\n";

      $config .= "# When someone tries to subscribe to a list, they have a\n";
      $config .= "# limited number of days within which they can confirm\n";
      $config .= "# their subscription.  2 or 3 days is about the minimum you\n";
      $config .= "# should set this to, since some people may not be able to\n";
      $config .= "# access their email immediately. 7 days is probably quite\n";
      $config .= "# excessive.\n\n";
      $config .= $this->constructConfigLine('DAYS_VALID',           $this->days);
      $config .= "\n\n";

      $config .= "# Set this variable to the maximum number of emails that\n";
      $config .= "# your host will let you send at any one time from a web\n";
      $config .= "# script.  For FutureQuest servers this value is currently\n";
      $config .= "# 500.  For infinite messages you can set this variable to\n";
      $config .= "# 0, but it is not recommended - you should instead set it\n";
      $config .= "# to a high number.\n\n";
      $config .= $this->constructConfigLine('MAX_EMAILS',           $this->maxemails);
      $config .= "\n\n";

      $config .= "# If you want to be 100% sure that the script is working\n";
      $config .= "# correctly, or if you want to keep records of every\n";
      $config .= "# confirmation email the script sends out, you should turn\n";
      $config .= "# this variable to \"on\".  It will send blind-carbon-copies\n";
      $config .= "# (BCC) of every email the script sends to the address you\n";
      $config .= "# listed above.\n\n";
      $config .= $this->constructConfigLine('EMAIL_COPIES_TO_SELF', $this->emailcopies);
      $config .= "\n\n";

      $config .= "# Testing mode turns off email sending and checking, so you\n";
      $config .= "# can test the script on a server that doesn't have email\n";
      $config .= "# services installed, such as my own development server....\n";
      $config .= "# Acceptable values are \"on\" and \"off\".\n\n";
      $config .= $this->constructConfigLine('TESTING_MODE',         $this->testingmode);
      $config .= "\n\n";

      $config .= "# Debug mode writes debugging information to a logfile that\n";
      $config .= "# you can send to namesuppressed support if the script isn't\n";
      $config .= "# working.  The logfile fills up quickly though, so you\n";
      $config .= "# should only turn it on when necessary.\n\n";
      $config .= $this->constructConfigLine('DEBUG_MODE',           $this->debugmode);
      $config .= "\n\n";

      $config .= "# Single Opt Out allows subscribers to unsubscribe from a\n";
      $config .= "# list without having to confirm the action.  This is the\n";
      $config .= "# preferred option - it allows people to unsubscribe an\n";
      $config .= "# email address even if they don't have access to it anymore.\n";
      $config .= "# Setting this to \"on\" will prevent many complaints.  Turning\n";
      $config .= "# this off will prevent people from unsubscribing addresses\n";
      $config .= "# without their permission, however.\n\n";
      $config .= $this->constructConfigLine('SINGLE_OPT_OUT',       $this->singleoptout);
      $config .= "\n\n";

      $config .= "# The banned domains and banned emails list lets you block\n";
      $config .= "# email addresses from ever subscribing to your mailing\n";
      $config .= "# lists.  It does not prevent those users from unsubscribing\n";
      $config .= "# if they have already subscribed.  When listing domains,\n";
      $config .= "# list the whole section after the @ sign... blocking a\n";
      $config .= "# domain will not necessariy block a subdomain.  Use a comma\n";
      $config .= "# to separate each domain or email address.\n\n";
      $config .= $this->constructConfigLine('BANNED_DOMAINS', implode(",", $this->banneddomains));
      $config .= $this->constructConfigLine('BANNED_EMAILS',  implode(",", $this->bannedemails));
      $config .= "\n\n\n";


      $config .= "###########################################################\n";
      $config .= "# Mailing list configuration                              #\n";
      $config .= "###########################################################\n";
      $config .= "#\n";
      $config .= "# Use the following entries as a template when setting up\n";
      $config .= "# your own lists.  Inside the brackets is the ID of your\n";
      $config .= "# mailing list, as it appears on the \"Mailing Lists\" page\n";
      $config .= "# of your FutureQuest CNC.  Be sure to get this right!\n";
      $config .= "#\n";
      $config .= "# Private refers to whether someone can subscribe to the\n";
      $config .= "# list without being invited.  Values are true/false.\n\n";

      foreach ($this->lists as $item) {
        $config .= $item->toString();
        $config .= "\n";
      }
      $config = trim($config);


      // Open the file in write mode, obliterating
      // the current file.  Danger danger danger!
      $fp = fopen($fn, 'w');
      if (!$fp) return false;

      // Quick, write out the remaining data and
      // then close the file.
      fwrite($fp, $config);
      fclose($fp);
      
      trigger_error("Leaving configuration->save.");
      return true;
    }

  }
?>