<?php

  //////////////////////////////////////////////////////////////
  ///
  ///  @class Emailer
  ///  @brief A generic emailing class.
  ///  @note  Copyright (c) 2005-2008 namesuppressed.
  ///
  ///  A generic emailing class (requires separate SMTP emailing
  ///  class to operate).  This class simplifies the task of
  ///  sending an email via SMTP or the built-in PHP email
  ///  functions.
  ///  
  ///  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
  ///  KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
  ///  WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
  ///  PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
  ///  OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
  ///  OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
  ///  OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
  ///  SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
  ///
  //////////////////////////////////////////////////////////////


  require_once('smtp.php');


  class Emailer {

    // SMTP mailout related variables
    var $smtp;         ///< The SMTP object we created in the constructor
    var $smtphost;     ///< The address/domain name of the SMTP host
    var $smtpusername; ///< The username we're sending email through
    var $smtppassword; ///< The password for the SMTP username
    var $smtprealm;    ///< The realm we use to send SMTP email
    var $smtpdebug;    ///< Output SMTP debugging messages?
    var $builtinmail;  ///< Use built-in mail functions instead of SMTP?

    var $sendername;   ///< Name of person sending emails (ie admin)
    var $senderemail;  ///< Email of person sending emails (ie admin)

    var $emailheader;  ///< useful global definition of email header
    var $emailcopies;  ///< set true to send copies to senderemail



    //////////////////////////////////////////////////////////////
    ///  Default constructor for the Emailer object class.
    //////////////////////////////////////////////////////////////

    function Emailer() {
    
      // Set some default values
      $this->setSenderName('Jane Q Public');
      $this->setSenderEmail('support@example.com');
      $this->setUseBuiltInMail(false);
      $this->setEmailCopiesToSender(false);

      // Set up the SMTP Class
      $this->smtp = new SMTPClass();
      $this->setSMTPHostname("localhost");   // relay SMTP server address
      $this->setSMTPLocalhost("localhost");  // Computer address
      $this->setSMTPUser('');                // Username
      $this->setSMTPPassword('');            // Password
      $this->setSMTPRealm('');               // Realm
      $this->setSMTPDebug(false);            // Turn off debugging
    }



    //////////////////////////////////////////////////////////////
    ///  Sets the name of the person sending the admin emails
    ///
    ///  @param   name  Name of the person sending admin emails
    //////////////////////////////////////////////////////////////

    function setSenderName($name) {
      $this->sendername = $name;
    }



    //////////////////////////////////////////////////////////////
    ///  Gets the name of the person sending the admin emails
    ///
    ///  @return  The name of the person sending admin emails
    //////////////////////////////////////////////////////////////

    function getSenderName() {
      return $this->sendername;
    }



    //////////////////////////////////////////////////////////////
    ///  Sets the email address that admin emails come from
    ///
    ///  @param email  Address of the person sending admin emails
    //////////////////////////////////////////////////////////////

    function setSenderEmail($email) {
      $this->senderemail  = $email;
    }


    //////////////////////////////////////////////////////////////
    ///  Gets the email address that admin emails come from
    ///
    ///  @return Email address of the person sending admin emails
    //////////////////////////////////////////////////////////////

    function getSenderEmail() {
      return $this->senderemail;
    }



    //////////////////////////////////////////////////////////////
    ///  Determines if the PHP mail routines should be used or the
    ///  additional SMTP classes.
    ///
    ///  @param bool  Boolean value, set true to use the built-in
    ///               PHP email routines.  Set false to use the
    ///               SMTP classes.
    //////////////////////////////////////////////////////////////

    function setUseBuiltInMail($bool) {
      $this->builtInMail = $bool;
    }



    //////////////////////////////////////////////////////////////
    ///  Gets the current value of the Built-In Mail flag.
    ///
    ///  @return  True if using the built-in PHP email routines,
    ///           false if using the SMTP Email class
    //////////////////////////////////////////////////////////////

    function getUseBuiltInMail() {
      return $this->builtInMail;
    }



    //////////////////////////////////////////////////////////////
    ///  Determines if copies of emails being sent should also be
    ///  sent to the sender (ie system administrator)
    ///
    ///  @param bool  Set true to email copies to sender, false
    ///               otherwise.
    //////////////////////////////////////////////////////////////

    function setEmailCopiesToSender($bool) {
      $this->emailcopies = $bool;
    }



    //////////////////////////////////////////////////////////////
    ///  Gets the current value of the Email Copies flag.
    ///
    ///  @return True if copies of emails are being sent to the
    ///          sender/admin address, false otherwise.
    //////////////////////////////////////////////////////////////

    function getEmailCopiesToSender() {
      return $this->emailcopies;
    }



    //////////////////////////////////////////////////////////////
    ///  Sets the SMTP Hostname to be used when sending email.
    ///  This is the relay SMTP server address
    ///
    ///  @param host  The SMTP Hostname to use
    //////////////////////////////////////////////////////////////

    function setSMTPHostname($host) {
      $this->smtp->host_name = $host;
    }



    //////////////////////////////////////////////////////////////
    ///  Gets the SMTP Hostname being used
    ///
    ///  @return The SMTP Hostname being used
    //////////////////////////////////////////////////////////////

    function getSMTPHostname() {
      return $this->smtp->host_name;
    }



    //////////////////////////////////////////////////////////////
    ///  Sets the internet name of the computer to send email
    ///  through.  Most commonly this should be set to localhost.
    ///
    ///  @param lh  The SMTP Localhost name to use
    //////////////////////////////////////////////////////////////

    function setSMTPLocalhost($lh) {
      $this->smtp->localhost = $lh;
    }
    
    
   
    //////////////////////////////////////////////////////////////
    ///  Gets the SMTP Localhost address/name
    ///
    ///  @return The name of the SMTP Localhost server
    //////////////////////////////////////////////////////////////

    function getSMTPLocalhost() {
      return $this->smtp->localhost;
    }
    
    
   
    //////////////////////////////////////////////////////////////
    ///  Sets the SMTP Username
    ///
    ///  @param user  The SMTP Username to use
    //////////////////////////////////////////////////////////////

    function setSMTPUser($user) {
      $this->smtp->user = $user;
    }
       
       
       
    //////////////////////////////////////////////////////////////
    ///  Gets the SMTP Username
    ///
    ///  @return The SMTP Username being used
    //////////////////////////////////////////////////////////////

    function getSMTPUser() {
      return $this->smtp->user;
    }
       
       
       
    //////////////////////////////////////////////////////////////
    ///  Sets the SMTP Password
    ///
    ///  @param pw  The SMTP Password to use
    //////////////////////////////////////////////////////////////

    function setSMTPPassword($pw) {
      $this->smtp->password = $pw;
    }

   
   
    //////////////////////////////////////////////////////////////
    ///  Gets the SMTP Password
    ///
    ///  @return The SMTP Password being used
    //////////////////////////////////////////////////////////////

    function getSMTPPassword() {
      return $this->smtp->password;
    }

   
   
    //////////////////////////////////////////////////////////////
    ///  Sets the SMTP Realm
    ///
    ///  @param realm  The SMTP Realm to use
    //////////////////////////////////////////////////////////////

    function setSMTPRealm($realm) {
      $this->smtp->realm = $realm;
    }



    //////////////////////////////////////////////////////////////
    ///  Gets the SMTP Realm
    ///
    ///  @return The SMTP Realm being used
    //////////////////////////////////////////////////////////////

    function getSMTPRealm() {
      return $this->smtp->realm;
    }



    //////////////////////////////////////////////////////////////
    ///  Enables and disables SMTP Debugging messages
    ///
    ///  @param enable  Set true to enable SMTP debugging, false
    ///                 to disable SMTP debugging.
    //////////////////////////////////////////////////////////////

    function setSMTPDebug($enable) {
      if ($enable) $this->smtp->debug = 1;
      else $this->smtp->debug = 0;
    }



    //////////////////////////////////////////////////////////////
    ///  Returns the value of the SMTP Debugging flag
    ///
    ///  @return True if SMTP Debugging is enabled, false otherwise
    //////////////////////////////////////////////////////////////

    function getSMTPDebug() {
      $value = $this->smtp->debug;
      if ($value == 0) return false;
      else if ($value == 1) return true;
      else return true;
    }



    //////////////////////////////////////////////////////////////
    ///  Sends out an email to a user
    ///
    ///  @param   a  email address to send email to
    ///  @param   subject  Subject Line of the email to send
    ///  @param   emailtext  The body of the email to send
    ///  @param   header  Any additional email headers to include
    ///  @return  true if the email was sent successfully
    ///           false otherwise
    //////////////////////////////////////////////////////////////

    function sendEmail($a, $subject, $emailtext, $header) {

      trigger_error('Entering emailer->sendEmail');

      // Check all of our inputs
      if (!is_string($a)) {
        trigger_error('Parameter a is not a string!');
        trigger_error("Can't send email to $a");
        trigger_error('Leaving emailer->sendEmail');
        return false;
      }
      if (empty($subject) || !is_string($subject)) {
        trigger_error('Subject line required - won\'t send email');
        trigger_error('Leaving emailer->sendEmail');
        return false;
      }
      if (empty($emailtext)) {
        trigger_error('Blank email message not allowed - won\'t send email');
        trigger_error('Leaving emailer->sendEmail');
        return false;
      }

      // Construct the email header template
      // Must use /r/n here see http://cr.yp.to/docs/smtplf.html
      $emailheader   = "Reply-To: " . $this->senderemail . "\r\n";
      $emailheader  .= "X-Mailer: PHP / ".phpversion()."\r\n";

      // Old style sending of email, forks sendmail everytime
      // and therefore not recommended (resource hog).  But
      // it's reliable, we can use it if the SMTP class fails.
      // Must use /r/n here see http://cr.yp.to/docs/smtplf.html
      if($this->builtinmail) {
        trigger_error('Using built-in PHP email system to send emails');
        $tempheader = $header;
        if ($emailcopies) $tempheader .= "Bcc: " . $this->senderemail . "\r\n";
        if (!mail($a, trim($subject), join("", $emailtext), $tempheader)) {
          trigger_error("Can't send email to $a.");
          trigger_error('Leaving emailer->sendEmail');
          return false;
        }
      }
      
      // Or we can use the SMTP class to send emails.
      else {
        trigger_error("Sending email to $a via SMTP");
        if(!$this->smtp->mail(
          $a,                              // Who the email is sent to
          '"' . $this->sendername . '" ' . // Address the email is sent from
          '<' . $this->senderemail . '>',
          $this->senderemail,              // Address to send bounced emails to
          trim($subject),                  // Subject line
          join("", $emailtext),            // The body as one concatenated string
          $emailheader)) {                 // The headers, concatenated with \r\n's
            trigger_error("Can't send email to $a.");
            trigger_error('Leaving emailer->sendEmail');
            return false;           // Would be better to return with $smtp->error
          }

        // Check if we also want to send email copies back to ourselves
        if ($this->emailcopies) {
          trigger_error('Attempting to send copy of email to ourselves');
          if(!$this->smtp->mail(
             $this->senderemail,              // Who the email is sent to
             '"' . $this->sendername . '" ' . // Address the email is sent from
             '<' . $this->senderemail . '>',
             $this->senderemail,              // Address to send bounced emails to
             trim($subject),                  // Subject line
             join("", $emailtext),            // The body as one concatenated string
             $emailheader)) {                 // The headers, concatenated with \r\n's
               trigger_error("Can't send email to $a.");
               trigger_error('Leaving emailer->sendEmail');
               return false;                  // Might be better to return with $smtp->error
             }
        }
      }
      trigger_error('Leaving emailer->sendEmail');
      return true;
    }

  }
?>