<?php

  //////////////////////////////////////////////////////////////
  ///
  ///  @class Ezmlm
  ///  @brief Engine / CLI / function layer
  ///  @note  Copyright (c) 2005-2008 namesuppressed.
  ///
  ///  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
  ///  KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
  ///  WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
  ///  PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
  ///  OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
  ///  OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
  ///  OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
  ///  SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
  ///
  //////////////////////////////////////////////////////////////




  require_once('configuration.php');
  require_once('wsemailer.php');
  require_once('logging.php');
  require_once('listmanager.php');
  require_once('transaction.php');


  class Ezmlm {

    var $config;       ///< Reference to Configuration class
    var $logging;      ///< Reference to Logging class
    var $listmanager;  ///< Reference to ListManager class
    var $wsemailer;    ///< Reference to WSEmailer class

    // Important configuration files and logs
    var $configfile;   ///< path to list configuration file



    //////////////////////////////////////////////////////////////
    ///  Constructor for the Ezmlm class, creates an object
    ///  representing the workings and configuration of a whole
    ///  mailing list engine.
    ///
    ///  @param  config  a configuration object
    ///  @return an Ezmlm object with the settings loaded from
    ///          the configuration file.
    //////////////////////////////////////////////////////////////

    function Ezmlm(&$config) {
      // Define constants for error messages
      define('ERR_OK', 0);
      define('ERR_ALREADY_SUBSCRIBED', 1);
      define('ERR_NOT_SUBSCRIBED', 2);
      define('ERR_NO_SUCH_FILE', 3);
      define('ERR_CANT_EMAIL', 4);
      define('ERR_CONFUSED', 5);
      define('ERR_INVALID_TOKEN', 6);
      define('ERR_FILE_NOT_WRITABLE', 7);
      define('ERR_NO_SUCH_TOKEN', 8);
      define('ERR_NO_SUCH_LIST', 9);
      define('ERR_CANT_SUBSCRIBE', 10);
      define('ERR_CANT_UNSUBSCRIBE', 11);
      define('ERR_INVALID_EMAIL', 12);
      define('ERR_BANNED_ADDRESS', 13);

      $this->config      = $config;
      $this->logging     = new Logging($config->basedir, $config->days);
      $this->listmanager = new ListManager($config->path, $config->listbase);
      $this->wsemailer   = new WSEmailer($config, $this->listmanager);

      // Add all lists to the ListManager
      $this->listmanager->setAllLists($config->lists);
    }




    //////////////////////////////////////////////////////////////
    ///  Returns a string explaining the meaning of a particular
    ///  error code generated by this class.
    ///
    ///  @param  errorcode  the code to translate into a string
    ///  @return the string description of the errorcode, or an
    ///          empty string if there is no description
    //////////////////////////////////////////////////////////////

    function errorToString($errorcode) {
      $errormsg = '';
      switch ($errorcode) {
        case ERR_OK:
          $errormsg = '';
          break;
        case ERR_ALREADY_SUBSCRIBED:
          $errormsg = 'You are already subscribed to that list!';
          break;
        case ERR_NOT_SUBSCRIBED:
          $errormsg = 'You are already unsubscribed from that list!';
          break;
        case ERR_NO_SUCH_FILE:
          $errormsg = 'Unable to write to WebScriber logfiles.';
          break;
        case ERR_CANT_EMAIL:
          $errormsg = 'Unable to send email.';
          break;
        case ERR_CONFUSED:
          $errormsg = 'I am confused, unable to interpret what you want.';
          break;
        case ERR_INVALID_TOKEN:
          $errormsg = 'The token is invalid.';
          break;
        case ERR_FILE_NOT_WRITABLE:
          $errormsg = 'Unable to write to file.';
          break;
        case ERR_NO_SUCH_TOKEN:
          $errormsg = 'There is no such token, sorry.';
          break;
        case ERR_NO_SUCH_LIST:
          $errormsg = 'There is no mailing list with that ID.';
          break;
        case ERR_CANT_SUBSCRIBE:
          $errormsg  = 'Problems subscribing address to ezMLM list. ';
          $errormsg .= 'Please contact list administrator for assistance.';
          break;
        case ERR_CANT_UNSUBSCRIBE:
          $errormsg  = 'Problems unsubscribing address from ezMLM list. ';
          $errormsg .= 'Please contact list administrator for assistance.';
          break;
        case ERR_INVALID_EMAIL:
          $errormsg  = 'The email address was invalid.';
          break;
        case ERR_BANNED_ADDRESS:
          $errormsg  = 'The email address you provided is not allowed to ';
          $errormsg .= 'subscribe to mailing lists here.  Please try another ';
          $errormsg .= 'email address from another service provider instead.';
          break;
        default:
          $errormsg = '';
      }
      return $errormsg;
    }




    //////////////////////////////////////////////////////////////
    /// PUBLIC API FUNCTION in beta testing
    ///
    /// @warning This function is still in beta testing!!
    //////////////////////////////////////////////////////////////

    function triggersubscribe($email, $listID) {
      trigger_error("Inside ezmlm->triggersubscribe.  Email: $email  listID: $listID");
      $trans   = new Transaction();
      $visitor = new Visitor();
      $visitor->fromEnvironment();
      $trans->setRequestVisitor($visitor);
      $trans->setEmail($this->listmanager->washAddress($email));
      trigger_error('Leaving ezmlm->triggersubscribe');
      return $this->subscribe($trans, $listID);
    }




    //////////////////////////////////////////////////////////////
    /// PUBLIC API FUNCTION in beta testing
    ///
    /// @warning This function is still in beta testing!!
    //////////////////////////////////////////////////////////////

    function triggerunsubscribe($email, $listID) {
      trigger_error("Inside ezmlm->triggerunsubscribe.  Email: $email  listID: $listID");
      $trans   = new Transaction();
      $visitor = new Visitor();
      $visitor->fromEnvironment();
      $trans->setRequestVisitor($visitor);
      $trans->setEmail($this->listmanager->washAddress($email));
      trigger_error('Leaving ezmlm->triggerunsubscribe');
      return $this->unsubscribe($trans, $listID);
    }




    //////////////////////////////////////////////////////////////
    /// Sends out a batch message to a single subscriber
    ///
    /// @param  trans     transaction object of the request
    /// @param  listname  listID of the list to subscribe to
    /// @return true if the message was sent successfully,
    ///         false otherwise
    //////////////////////////////////////////////////////////////

    function batch($trans, $listname) {
      return $this->unifiedactions('batch', $trans, $listname);
    }




    //////////////////////////////////////////////////////////////
    /// Sends a subscribe email message to a potential subscriber
    ///
    /// @param  trans     transaction object of the request
    /// @param  listname  listID of the list to subscribe to
    /// @return true if the subscription was successful
    //////////////////////////////////////////////////////////////

    function subscribe($trans, $listname) {
      return $this->unifiedactions('subscribe', $trans, $listname);
    }




    //////////////////////////////////////////////////////////////
    /// Sends an unsubscribe email to a single subscriber
    ///
    /// @param  trans     transaction object of the request
    /// @param  listname  listID of the list to unsubscribe from
    /// @return true if the unsubscription was successful
    //////////////////////////////////////////////////////////////

    function unsubscribe($trans, $listname) {
      return $this->unifiedactions('unsubscribe', $trans, $listname);
    }




    //////////////////////////////////////////////////////////////
    /// Allows a subscriber to update their subscriptions
    ///
    /// @param  trans  transaction object of the request
    /// @return true if the updates were successful, else false
    //////////////////////////////////////////////////////////////

    function update($trans) {
      return $this->unifiedactions('update', $trans, null);
    }




    //////////////////////////////////////////////////////////////
    ///  Checks if an email address is banned from subscribing to
    ///  any mailing lists.
    ///
    ///  @param  address  the email to check if banned
    ///  @param  domains  array of banned domains
    ///  @param  emails   array of banned emails
    ///  @return true if address is banned, false otherwise
    //////////////////////////////////////////////////////////////

    function isBanned($address, $domains, $emails) {
      $domains = array_map('trim', $domains);
      $emails  = array_map('trim', $emails);
      $domains = array_map('strtolower', $domains);
      $emails  = array_map('strtolower', $emails);
      $index = strpos($address, '@');
      if ($index === false) return false;
      if ($index == (strlen($address) - 1)) return true;
      $domain = substr($address, $index + 1);
      $banneddomain = in_array(strtolower($domain),  $domains);
      $bannedaddy   = in_array(strtolower($address), $emails);
      return ($banneddomain || $bannedaddy);
    }




    //////////////////////////////////////////////////////////////
    ///  Performs a subscription, unsubscription, or sends out a
    ///  bulk subscription notification depending on the value of
    ///  the $action variable.  The variable can take the values
    ///  of subscribe, unsubscribe, batch, or update.
    ///
    ///  @param   action  batch/subscribe/unsubscribe/update
    ///  @param   transaction  the transaction object to use
    ///  @param   listname  listID of the list to act on
    ///  @return  ERR_OK if there were no errors, another
    ///           predefined error code otherwise.
    //////////////////////////////////////////////////////////////

    function unifiedactions($action, $transaction, $listname) {

      trigger_error('Inside ezmlm->unifiedactions.');
      trigger_error("Action: $action, Listname: $listname");

      // Check that it's a valid email address
      if (!$this->listmanager->isValidEmail($transaction->getEmail())) {
        trigger_error('Invalid email address');
        return ERR_INVALID_EMAIL;
      }

      if ($action != 'update' && $action != 'batch' &&
          $action != 'subscribe' && $action != 'unsubscribe') {
        // Return some kind of error, we can't do anything
        trigger_error("Hmm, I'm confused.");
        trigger_error('Leaving ezmlm->unifiedactions.');
        return ERR_CONFUSED;
      }


      // Perform a number of checks and actions, but only if
      // we're not performing an update.
      if ($action != 'update') {

        // Check that the list exists
        if (!$this->listmanager->listExists($listname)) {
          trigger_error("No such list as $listname");
          trigger_error('Leaving ezmlm->unifiedactions.');
          return ERR_NO_SUCH_LIST;
        }

        // Construct the transaction object
        if ($action == 'subscribe' || $action == 'batch')
          $transaction->subscribeList($listname);
        if ($action == 'unsubscribe')
          $transaction->unsubscribeList($listname);
        $transaction->generateRequestTimestamp();
        $transaction->generateToken();

        // Check if the user is subscribed/not subscribed
        $issubscribed = $this->listmanager->isSubscribed($transaction->getEmail(), $listname);
        if ($action == 'subscribe' || $action == 'batch') {
          if ($issubscribed) {
            trigger_error('Address is already subscribed');
            trigger_error('Leaving ezmlm->unifiedactions.');
            return ERR_ALREADY_SUBSCRIBED;
          }
          $domains = $this->config->banneddomains;
          $emails  = $this->config->bannedemails;
          $isbanned = $this->isBanned($transaction->getEmail(), $domains, $emails);
          if ($isbanned) {
            trigger_error('Address is banned from subscribing');
            trigger_error('Leaving ezmlm->unifiedactions.');
            return ERR_BANNED_ADDRESS;
          }
        }
        if ($action == 'unsubscribe') {
          if (!$issubscribed) {
            trigger_error('Address is not subscribed');
            trigger_error('Leaving ezmlm->unifiedactions.');
            return ERR_NOT_SUBSCRIBED;
          }
        }
      }
      else {
        // We're updating, so if a banned user tries to subscribe
        // via the update mechanism we need to block them.
        $domains  = $this->config->banneddomains;
        $emails   = $this->config->bannedemails;
        $isbanned = $this->isBanned($transaction->getEmail(), $domains, $emails);
        if (count($transaction->getToSubscribe()) > 0 && $isbanned) {
          trigger_error('Address is banned from subscribing');
          trigger_error('Leaving ezmlm->unifiedactions.');
          return ERR_BANNED_ADDRESS;
        }
      }


      // Use single opt-out if we're only unsubscribing
      if ($this->config->singleoptout &&
          count($transaction->getToSubscribe()) == 0) {
        trigger_error('Single Opt Out is enabled');
        $visitor = $transaction->getRequestVisitor();
        $transaction->setConfirmVisitor($visitor);
        trigger_error('Leaving ezmlm->unifiedactions.');
        return $this->confirmTransaction($transaction);
      }


      // Otherwise, add transaction to the pending logfile.
      $okay = $this->logging->addToPending($transaction);
      if (!$okay) {
        trigger_error('Leaving ezmlm->unifiedactions.');
        return ERR_NO_SUCH_FILE;
      }


      // Send the emails out
      if ($action == 'batch') {
        $okay = $this->wsemailer->sendBatchEmail($transaction);
        if (!$okay) return ERR_CANT_EMAIL;
      }
      else if ($action == 'subscribe') {
        $okay = $this->wsemailer->sendSubscribeEmail($transaction);
        if (!$okay) return ERR_CANT_EMAIL;
      }
      else if ($action == 'unsubscribe') {
        $okay = $this->wsemailer->sendUnsubscribeEmail($transaction);
        if (!$okay) return ERR_CANT_EMAIL;
      }
      else if ($action == 'update') {
        $okay = $this->wsemailer->sendUpdateEmail($transaction);
        if (!$okay) return ERR_CANT_EMAIL;
      }


      // Everything okay, so just return okay.
      trigger_error('Leaving ezmlm->unifiedactions.');
      return ERR_OK;
    }




    //////////////////////////////////////////////////////////////
    ///  Confirm a subscribe/unsubscribe action by confirming a
    ///  token value that should exist in the Pending logfile.
    ///
    ///  @param   token  the token value to confirm
    ///  @param   visitor  transaction object of the visitor
    ///           confirming this token.
    ///  @return  ERR_OK if there were no errors, another
    ///           predefined error code otherwise.
    //////////////////////////////////////////////////////////////

    function confirm($token, $visitor) {
      $trans = $this->logging->getPending($token);
      if ($trans == false) {
        trigger_error('No such token!');
        return ERR_NO_SUCH_TOKEN;
      }
      trigger_error('Token found.');
      $trans->setConfirmVisitor($visitor);
      return $this->confirmTransaction($trans);
    }




    //////////////////////////////////////////////////////////////
    ///  Confirm a subscribe/unsubscribe action.
    ///
    ///  @param   trans  the transaction to confirm
    ///  @return  ERR_OK if there were no errors, another
    ///           predefined error code otherwise.
    //////////////////////////////////////////////////////////////

    function confirmTransaction($trans) {

      trigger_error('Inside ezmlm->confirmTransaction');

      $p  = $this->config->path;
      $lb = $this->config->listbase;

      if (!$trans->isValidToken($this->config->days)) {
        trigger_error('Token is invalid.');
        trigger_error('Leaving ezmlm->confirmTransaction.');
        return ERR_INVALID_TOKEN;
      }

      // Now we have a valid token, lets wash the listname and
      // the address ready for use
      $a  = $trans->getEmail();
      $subslist  = $trans->getToSubscribe();
      $unsublist = $trans->getToUnsubscribe();

      // Make changes to Transaction
      $trans->setEmail($this->listmanager->washAddress($a));

      // If we're performing more than one subscription/unsubscription
      // we must be performing an update request.
      $okay   = false;
      $subsok = false;
      trigger_error('Starting to update subscriptions');
      if ($trans->getNumTasks() > 1) {
        for ($i = 0; $i < count($subslist); $i++) {
          $subsok |= $this->listmanager->subscribe($a, $subslist[$i]);
          if (!$subsok) {
            trigger_error('Problems with subscribing');
            trigger_error('Leaving ezmlm->confirmTransaction');
            return ERR_CANT_SUBSCRIBE;
          }
        }
        for ($i = 0; $i < count($unsublist); $i++) {
          $subsok |= $this->listmanager->unsubscribe($a, $unsublist[$i]);
          if (!$subsok) {
            trigger_error('Problems with unsubscribing');
            trigger_error('Leaving ezmlm->confirmTransaction');
            return ERR_CANT_UNSUBSCRIBE;
          }
        }
        if ($subsok) {
          $okay = $this->wsemailer->sendUpdateConfirmEmail($trans);
        }
      }
      else {
        if (count($subslist) == 1) {
          $subsok = $this->listmanager->subscribe($a, $subslist[0]);
          if (!$subsok) {
            trigger_error('Problems with subscribing');
            trigger_error('Leaving ezmlm->confirmTransaction');
            return ERR_CANT_SUBSCRIBE;
          }
          else {
            $okay = $this->wsemailer->sendSubscribeConfirmEmail($trans);
          }
        }
        if (count($unsublist) == 1) {
          $subsok = $this->listmanager->unsubscribe($a, $unsublist[0]);
          if (!$subsok) {
            trigger_error('Problems with unsubscribing');
            trigger_error('Leaving ezmlm->confirmTransaction');
            return ERR_CANT_UNSUBSCRIBE;
          }
          else {
            $okay = $this->wsemailer->sendUnsubConfirmEmail($trans);
          }
        }
      }
      trigger_error('Finished updating subscriptions');


      // If the emails couldn't be send, return with an error
      // and don't confirm it in the logfiles just yet.
      if (!$okay) {
        trigger_error('Leaving ezmlm->confirmTransaction');
        return ERR_CANT_EMAIL;
      }

      // Log this confirmation to the, uhm, confirmation logs
      $this->logging->addToConfirmed($trans);

      trigger_error('Leaving ezmlm->confirmTransaction');
      return ERR_OK;
    }




    //////////////////////////////////////////////////////////////
    ///  Parses a string (or an array of strings), replacing a set
    ///  of codes in the strings with their corresponding values.
    ///  The codes and values must be passed as an array.  The
    ///  codes is the array key, and what the code is to be
    ///  replaced with is the array result.  In the strings,
    ///  codes to be replaced must be surrounded with double
    ///  ampersands before and after them.  Here's an example:
    ///
    ///  @code
    ///    $document = '&&TEST&&';
    ///    $codes = array();
    ///    $codes['TEST'] = 'This was replaced by parse.';
    ///    parse($document, $codes);
    ///    echo $document;
    ///  @endcode
    ///
    ///  @param arraystring  an array of strings to parse
    ///  @param codes  an array of codes to be replaced
    ///  @return an array of parsed strings
    //////////////////////////////////////////////////////////////

    function parser($arraystring, $codes) {

      // Set up all the tags to be replaced
      $search = array();
      $replace = array();
      foreach ($codes as $key => $result) {
        $code = '/&&' . $key . '&&/i';
        array_push($search, $code);
        array_push($replace, $result);
      }

      // And now we actually perform the replace
      $arraystring = preg_replace($search, $replace, $arraystring);
      return $arraystring;
    }




    //////////////////////////////////////////////////////////////
    ///  Parses a string (or an array of strings), replacing a set
    ///  of predefined codes with their corresponding values.  The
    ///  values are drawn from the given transaction object,
    ///  listname, token and timevalue.
    ///
    ///  @param   arraystring  an array of strings to parse
    ///  @param   transaction  the current transaction object
    ///  @param   error  an error message to include in output
    ///  @return  an array of parsed strings
    //////////////////////////////////////////////////////////////

    function parse($arraystring, $transaction, $error) {

      trigger_error('Inside ezmlm->parse');

      $action   = '';
      $tasklist = '';
      $maillist = false;
      $numtasks = $transaction->getNumTasks();

      // Compute the replacement for &&TASKLIST&&
      if ($numtasks > 0) {
        for ($i = 0; $i < $numtasks; $i++) {
          $a = $transaction->getTask($i);
          $l = $this->listmanager->getList($a->listname);
          if ($l != false) {
            if ($i > 0) $tasklist .= "\n\n";
            $tasklist .= 'List Name: ';
            $tasklist .= $l->getName() . "\n";
            $tasklist .= 'Action: ';
            if ($a->action == 's') $tasklist .= 'Subscribe';
            if ($a->action == 'u') $tasklist .= 'Remove';
          }
          else trigger_error("No such list: $a->listname");
        }
      }

      // Decide what to put in the action field and
      // what to use as our maillist (if anything)
      if ($numtasks > 0) {
        if ($numtasks > 1) {
          $action = 'multiple';
          $maillist = false;
        }
        else {
          $a = $transaction->getTask(0);
          if ($a->action == 's') $action .= 'subscribe';
          if ($a->action == 'u') $action .= 'unsubscribe';
          $maillist = $this->listmanager->getList($a->listname);
        }
      }

      $codes = array();
      $codes['ACTION'] = $action;
      $codes['CONFIRMURL']  = $this->config->scripturl . "?t=" . $transaction->getToken();
      $codes['DATESTAMP']   = date("r", $transaction->requestvisitor->getTimestamp());
      $codes['EMAIL']       = $transaction->getEmail();
      $codes['ERROR']       = $error;
      $codes['EXPIRYDAYS']  = $this->config->days;
      $codes['IPADDRESS']   = $transaction->requestvisitor->getIP();
      if (isset($maillist) && $maillist != false) {
        $codes['LISTDESCRIPTION'] = $maillist->getDescription();
        $codes['LISTNAME']        = $maillist->getName();
        $codes['LISTSUBSCRIBE']   = $maillist->subscribe;
        $codes['LISTUNSUBSCRIBE'] = $maillist->unsubscribe;
      }
      $codes['NOTE']        = $transaction->getNote();
      $codes['SCRIPTURL']   = $this->config->scripturl;
      $codes['SENDEREMAIL'] = $this->config->senderemail;
      $codes['SENDERNAME']  = $this->config->sendername;
      $codes['TASKLIST']    = $tasklist;
      $codes['USERAGENT']   = $transaction->requestvisitor->getUseragent();

      $arrayresult = $this->parser($arraystring, $codes);
      trigger_error('Leaving ezmlm->parse');
      return $arrayresult;
    }
  }
?>