<?php

  //////////////////////////////////////////////////////////////
  ///
  ///  @class ListManager
  ///  @brief Manages lists and subscriptions.
  ///  @note  Copyright (c) 2005-2008 namesuppressed.
  ///
  ///  This class should only include functions related to
  ///  managing mailing lists - subscribing and unsubscribing
  ///  addresses, getting a list of available mailing lists and
  ///  so on.  Filtering should NOT occur in this class, it
  ///  should occur in the class that calls this one.
  ///
  ///  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
  ///  KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
  ///  WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
  ///  PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
  ///  OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
  ///  OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
  ///  OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
  ///  SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
  ///
  //////////////////////////////////////////////////////////////


  require_once('maillist.php');


  class ListManager {

    var $lists;         ///< array of mailing list objects
		var $path;          ///< path to ezmlm functions
		var $listbase;      ///< path where ezmlm directories are stored


    //////////////////////////////////////////////////////////////
    ///  Default constructor for the ListManager class.
    ///
    ///  @param path      path to ezmlm binary files/functions
    ///  @param listbase  path to ezmlm directories on computer
    ///  @return  A ListManager object that you can call to run
    ///           the functions contained within it.
    //////////////////////////////////////////////////////////////

    function ListManager($path, $listbase) {
      $this->lists    = array();
      $this->path     = $path;
      $this->listbase = $listbase;
    }



    //////////////////////////////////////////////////////////////
    ///  Adds a mailing list to the array of mailing lists that
    ///  this ListManager handles.
    ///
    ///  @param   listID     a string 
    ///  @param   name       a string
    ///  @param   desc       a string
    ///  @param   isPrivate  a boolean
    ///  @param   subsEmail  a string
    ///  @param   unsubEmail a string
    ///  @return  A ListManager object that you can call to run
    ///           the functions contained within it.
    //////////////////////////////////////////////////////////////

    function addList($listID, $name, $desc, $isPrivate,
                     $subsEmail, $unsubEmail) {
			$listitem = new Maillist();
			$listitem->setID($listID);
			$listitem->setName($name);
			$listitem->setDescription($desc);
			$listitem->setPrivate($isPrivate);
			$listitem->setSubscribeEmail($subsEmail);
			$listitem->setUnsubscribeEmail($unsubEmail);
			array_push($this->lists, $listitem);
    }



    //////////////////////////////////////////////////////////////
    ///  Sets the entire array of mailing lists
    ///
    ///  @param   array  an array of Maillist objects
    //////////////////////////////////////////////////////////////

    function setAllLists($array) {
			$this->lists = $array;
    }



		//////////////////////////////////////////////////////////////
		///  Creates and returns a mailing list object of the
		///  requested mailing list.
		///
		///  @param  listID  the ID of the list to retrieve
		///  @return a List object of the requested mailing list
		///          or false if the list cannot be found.
		//////////////////////////////////////////////////////////////

		function getList($listID) {
			trigger_error("Inside listmanager->getList.  ListID: $listID");
			foreach ($this->lists as $item)
				if ($item->getID() == $listID) return $item;
			trigger_error("Leaving listmanager->getList.");
			return false;
		}



    //////////////////////////////////////////////////////////////
    ///  Remove a list from the array of lists
    ///
    ///  @todo    This function has not been implemented
    ///  @param   listID   a string
    ///  @return  A ListManager object that you can call to run
    ///           the functions contained within it.
    //////////////////////////////////////////////////////////////

    function removeList($listID) {
      // Code to remove the list from the array
    }



		//////////////////////////////////////////////////////////////
		///  Returns an array of Maillist objects of every mailing list
		///  available under the current Webscriber configuration.
		///
		///  @return an array of every list in this configuration
		//////////////////////////////////////////////////////////////

		function getAllLists() {
      return $this->lists;
		}



		//////////////////////////////////////////////////////////////
		///  Returns an array of Maillist objects of every public
		///  mailing list in the current Webscriber configuration.
		///
		///  @return an array of every public mailing list
		//////////////////////////////////////////////////////////////

		function getPublicLists() {
			trigger_error("Inside listmanager->getPublicLists.");
			$publicLists = array();
			foreach ($this->lists as $item)
				if (!$item->isPrivate()) array_push($publicLists, $item);
			trigger_error("Leaving listmanager->getPublicLists.");
			return $publicLists;
		}



		//////////////////////////////////////////////////////////////
		///  Returns an array of Maillist objects of every private
		///  mailing list in the current Webscriber configuration.
		///
		///  @return  an array of every private mailing list
		//////////////////////////////////////////////////////////////

		function getPrivateLists() {
			trigger_error("Inside listmanager->getPrivateLists.");
			$privateLists = array();
			foreach ($this->lists as $item)
				if ($item->isPrivate()) array_push($privateLists, $item);
			trigger_error("Leaving listmanager->getPrivateLists.");
			return $privateLists;
		}



		//////////////////////////////////////////////////////////////
		///  Subscribes an email address to a mailing list.
		///
		///  @param  list   ID of the list to get addresses from
		///  @param  email  email address to subscribe
		///  @return true if successful, false otherwise
		//////////////////////////////////////////////////////////////

		function subscribe($email, $list) {
			trigger_error("Inside listmanager->subscribe.  List: $list  Email: $email");
			$p  = $this->path;      // We should wash this
			$lb = $this->listbase;  // We should wash this
			$a  = $this->washAddress($email);
			$ln = $this->washListname($list);
			$command = $p."ezmlm-sub ".$lb.$ln." ".$a;
			trigger_error("Running $command");
			exec($command);
			if (!$this->isSubscribed($email, $list)) {
				trigger_error("Can't subscribe $a to $ln.");
				trigger_error('Leaving listmanager->subscribe.');
				return false;
			}
			trigger_error("Leaving listmanager->subscribe.");
			return true;
		}



		//////////////////////////////////////////////////////////////
		///  Unsubscribes an email address from a mailing list.
		///
		///  @param  email  email address to subscribe
		///  @param  list   ID of the list to unsubscribe from
		///  @return true if successful, false otherwise
		//////////////////////////////////////////////////////////////

		function unsubscribe($email, $list) {
			trigger_error("Inside listmanager->unsubscribe.  List: $list  Email: $email");
			$p  = $this->path;      // We should wash this
			$lb = $this->listbase;  // We should wash this
			$a  = $this->washAddress($email);
			$ln = $this->washListname($list);
			$command = $p."ezmlm-unsub ".$lb.$ln." ".$a;
			trigger_error("Running $command");
			exec($command);
			if ($this->isSubscribed($email, $list)) {
				trigger_error("Can't unsubscribe $a to $ln.");
				trigger_error('Leaving listmanager->unsubscribe.');
				return false;
			}
			trigger_error("Leaving listmanager->unsubscribe.");
			return true;
		}



		//////////////////////////////////////////////////////////////
		///  Returns an array containing a list of subscribers to a
		///  given mailing list.
		///
		///  @param  list  ID of the list to get addresses from
		///  @return an array of subscriber addresses
		//////////////////////////////////////////////////////////////

		function getSubscriberList($list) {
			trigger_error("Inside listmanager->getSubscriberList.  List: $list");
			$addresses = array();
			$p  = $this->path;
			$lb = $this->listbase;
			$ln = $this->washListname($list);
			$command = $p."ezmlm-list ".$lb.$ln;
			if ($this->listExists($ln)) {
				trigger_error("Running $command");
				exec($command, $addresses);
			}
			trigger_error("Leaving listmanager->getSubscriberList.");
			return $addresses;
		}



		//////////////////////////////////////////////////////////////
		///  Checks if an email address is subscribed to a particular
		///  mailing list.
		///
		///  @param  address  the email to check if subscribed
		///  @param  list     the listID of the list to check
		///  @return true if address is subscribed to the list
		//////////////////////////////////////////////////////////////

		function isSubscribed($address, $list) {
			$addresses = $this->getSubscriberList($list);
			$addresses = array_map("strtolower", $addresses);
			return in_array(strtolower($address), $addresses);
		}



		//////////////////////////////////////////////////////////////
		///  Checks for the pre-existence of a mailing list - that is,
		///  it checks if the list is in the configuration files.  It
		///  should also check that the given ezMLM mailing list
		///  actually exists, but we currently have no way to check.
		///
		///  @todo   add check that ezmlm list actually exists
		///  @param  ln  the listID of the list to check
		///  @return true if the list exists, false otherwise
		//////////////////////////////////////////////////////////////

		function listExists($ln) {
			$listname = $this->washListname($ln);
			$lists    = $this->getAllLists();
			foreach ($lists as $list) {
				if ($list->getID() == $listname) return true;
			}		
			return false;
		}




		//////////////////////////////////////////////////////////////
		///  Checks if an email address is in a valid format.  This
		///  function does not check email addresses against the RFC
		///  standard, rather against email's common usage.  This
		///  function does not check if the email address is an active
		///  working inbox however.
		///
		///  @param  address  the address to check for validity
		///  @return true if a valid email format, false otherwise
		//////////////////////////////////////////////////////////////

		function isValidEmail($address) {
			$address  = trim($address);
			if (empty($address)) return false;
			$email_nr = eregi_replace("\n", "", $address);
			$email    = eregi_replace(" +", "", $email_nr);
			if(!eregi("^([a-zA-Z0-9_\.\-])+\@".
								"(([a-zA-Z0-9\-])+\.)+([a-zA-Z0-9]+)+$", $email))
				return false;
			return true;
		}



		//////////////////////////////////////////////////////////////
		///  "Washes" an email address so that it can be safely
		///  used (ie no pipes, backticks, brackets, spaces)
		///
		///  @todo   Replace strtok call with strpos
		///  @param  address  the email address to wash
		///  @return the washed email address
		//////////////////////////////////////////////////////////////

		function washAddress($address) {
			$washedaddress = strtok($address, " ");
			$washedaddress = preg_replace("/ /",  "", $washedaddress);
			$washedaddress = preg_replace("/\n/", "", $washedaddress);
			$washedaddress = preg_replace("/\r/", "", $washedaddress);
			$washedaddress = preg_replace("/\|/", "", $washedaddress);
			$washedaddress = preg_replace("/\</", "", $washedaddress);
			$washedaddress = preg_replace("/\>/", "", $washedaddress);
			$washedaddress = preg_replace("/\(/", "", $washedaddress);
			$washedaddress = preg_replace("/\)/", "", $washedaddress);
			$washedaddress = preg_replace("/\`/", "", $washedaddress);
			$washedaddress = strtolower($washedaddress);
			return $washedaddress;
		}



		//////////////////////////////////////////////////////////////
		///  "Washes" a listname so that it can be safely used in calls
		///  calls to ezMLM command line programs (so no pipes, spaces
		///  backticks or brackets are allowed).
		///
		///  @todo   Replace strtok call with strpos
		///  @param  listname  the listname to wash
		///  @return the washed listname
		//////////////////////////////////////////////////////////////

		function washListname($listname) {
			$washedlistname = strtok($listname, " ");
			$washedlistname = preg_replace("/\|/", "", $washedlistname);
			$washedlistname = preg_replace("/\</", "", $washedlistname);
			$washedlistname = preg_replace("/\>/", "", $washedlistname);
			$washedlistname = preg_replace("/\(/", "", $washedlistname);
			$washedlistname = preg_replace("/\)/", "", $washedlistname);
			$washedlistname = preg_replace("/\`/", "", $washedlistname);
			return $washedlistname;
		}

  }
?>