<?php

  //////////////////////////////////////////////////////////////
  ///
  ///  @class Logging
  ///  @brief Handles logging of subscriptions data
  ///  @note  Copyright (c) 2005-2008 namesuppressed.
  ///
  ///  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
  ///  KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
  ///  WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
  ///  PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
  ///  OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
  ///  OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
  ///  OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
  ///  SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
  ///
  //////////////////////////////////////////////////////////////



  require_once('transaction.php');

  class Logging {

	// Log file locations
    var $logpending;   ///< path to log file of pending requests
    var $logconfirmed; ///< path to log file of completed requests
    var $logprogress;  ///< path to log file of requests in progress
    var $logdebug;     ///< path to log file of debug messages

    var $days;         ///< number of days that a token is valid for



    //////////////////////////////////////////////////////////////
    ///  Default constructor for the Logging object class.
    ///
    ///  @param   basedir
    ///             The directory where the log files are stored.
    ///  @param   d Number of days that a pending transaction is
    ///             valid for
    ///  @return  A Configuration object
    //////////////////////////////////////////////////////////////

    function Logging($basedir, $d) {
	  $this->logpending   = $basedir . 'logs/pending';
	  $this->logprogress  = $basedir . 'logs/progress';
	  $this->logconfirmed = $basedir . 'logs/confirmed';
	  $this->logdebug     = $basedir . 'logs/debug';
	  $this->days         = $d;
    }



    //////////////////////////////////////////////////////////////
    ///  Tests if the logfiles can be read and written to.
    ///
    ///  @return  true if the logfiles are readable and writable,
    ///           false otherwise
    //////////////////////////////////////////////////////////////

    function testFiles() {
      $logfilesok  = is_readable($this->logpending);
      $logfilesok &= is_writable($this->logpending);
	  $logfilesok &= is_readable($this->logconfirmed);
	  $logfilesok &= is_writable($this->logconfirmed);
	  $logfilesok &= is_readable($this->logprogress);
	  $logfilesok &= is_writable($this->logprogress);
	  $logfilesok &= is_readable($this->logdebug);
      $logfilesok &= is_writable($this->logdebug);
      return $logfilesok;
    }



	//////////////////////////////////////////////////////////////
	///  Returns an array of log files that couldn't be opened
	///
	///  @return Array of strings of missing classes
	//////////////////////////////////////////////////////////////

	function getMissingLogFiles() {
	  $logs = array($this->logpending, $this->logconfirmed, $this->logprogress, $this->logdebug);
		$missing = array();
		foreach ($logs as $item) {
			if (!is_readable($item) && !is_writable($item)) array_push($missing, $item);
		}
		return $missing;
	}



		//////////////////////////////////////////////////////////////
		///  Add a pending transaction to the pending logfile.
		///
		///  @param  trans  the transaction to store in pending logs
		///  @return true if successful, false otherwise
		//////////////////////////////////////////////////////////////

		function addToPending($trans) {
			trigger_error("Inside logging->addToPending.");
			$logentry = $trans->toString($trans);
			$fp = fopen($this->logpending, "a");
			if ($fp) {
				$string_to_write = $logentry . "\n";
				fwrite($fp, $string_to_write);
				fclose($fp);
			  trigger_error("Leaving logging->addToPending.");
				return true;
			}
			else {
		  	trigger_error("Unable to load logpending: $this->logpending");
			  trigger_error("Leaving logging->addToPending.");
			  return false;
			}
		}



		//////////////////////////////////////////////////////////////
		///  Get a pending transaction from the pending logfile.
		///
		///  @param  token  the token to get details about
		///  @return a Transaction object with details of the
		///          transaction with given token, or false if no
		///          such token found in the pending logs.
		//////////////////////////////////////////////////////////////

		function getPending($token) {

			trigger_error("Inside logging->getPending.  Token: $token");
      trigger_error("Looking for tokens matching $token in pending logfile");

			// Get all lines beginning with token and store in $matches
			$matches = array();
			$remainder = array();
			$matchstring = "/^" . $token . "\|.*/";
			foreach (file($this->logpending) as $line) {
				if (preg_match($matchstring, $line)) array_push($matches, $line);
				else array_push($remainder, $line);
			}
			if (sizeof($matches) < 1) {
        trigger_error('No matches, unable to find transaction in pending logs.');
				trigger_error('Leaving logging->getPending.');
				return false;
			}

			$trans = new Transaction();
			foreach ($matches as $line) {
				$trans->fromString($line);
			}
			trigger_error("Leaving logging->getPending.");
			return $trans;
		}



    //////////////////////////////////////////////////////////////
    ///  Takes a transaction, adds it to the confirmation logs
    ///  and then removes the corresponding transaction (with
    ///  the matching token) from the pending logs.
    ///
    ///  @param  trans  the transaction object to confirm
    ///  @return true if the confirmation was logged okay, false
    ///          if there was a problem writing to a logfile.
    //////////////////////////////////////////////////////////////

    function addToConfirmed($trans) {

      // Append the token line to the confirmed log
      trigger_error('Inside logging->addToConfirmed.');
      trigger_error('Appending token to confirmed log');
      if (is_writable($this->logconfirmed)) {
        if (!$fp = fopen($this->logconfirmed, 'a')) {
          trigger_error("Can't open $this->logconfirmed.");
          trigger_error('Leaving logging->addToConfirmed.');
          return false;
        }
        fwrite($fp, $trans->toString() . "\n");
        fclose($fp);
      }
      else {
        trigger_error("File not writable: $this->logconfirmed.");
        trigger_error('Leaving logging->addToConfirmed.');
        return false;
      }

      // Write out the pending logfile again, minus any
      // transations with the token that was used just now.
      trigger_error('Rewriting pending logfile');
			$matches = array();
			$remainder = array();
			$matchstring = "/^" . $trans->getToken() . "\|.*/";
			foreach (file($this->logpending) as $line) {
				if (preg_match($matchstring, $line)) array_push($matches, $line);
				else array_push($remainder, $line);
			}
      if (is_writable($this->logpending)) {
        if (!$fp = fopen($this->logpending, 'w')) {
          trigger_error("Can't open $this->logpending.");
          trigger_error('Leaving logging->addToConfirmed.');
          return false;
        }
        fwrite($fp, join("",$remainder));
        fclose($fp);
      }
      else {
        trigger_error("File not writable: $this->logpending.");
        trigger_error('Leaving logging->addToConfirmed.');
        return false;
      }
      
      return true;
    }



		//////////////////////////////////////////////////////////////
		///  Confirm a subscribe/unsubscribe action by confirming a
		///  token value that should exist in the confirmed logfile.
		///
		///  @param  token  the token to get details about
		///  @return a Transaction object with details of the
		///          transaction with given token, or false if no
		///          such token found in confirmed logs.
		//////////////////////////////////////////////////////////////

		function getConfirmed($token) {

			trigger_error("Inside logging->getConfirmed.  Token: $token");

			// Get all lines beginning with token and store in $matches
			$matches = array();
			$remainder = array();
			$matchstring = "/^" . $token . "\|.*/";
			foreach (file($this->logconfirmed) as $line) {
				if (preg_match($matchstring, $line)) array_push($matches, $line);
				else array_push($remainder, $line);
			}
			if (sizeof($matches) < 1) {
				trigger_error("No matches found.");
        trigger_error('Unable to find transaction in confirmed logs.');
				trigger_error("Leaving logging->getConfirmed.");
				return false;
			}

			$trans = new Transaction();
			foreach ($matches as $line) {
				$trans->fromString($line);
			}
			trigger_error("Leaving logging->getConfirmed.");
			return $trans;
		}



    //////////////////////////////////////////////////////////////
    ///  Updates the pending and progress logs by removing entries
    ///  that are out of date or damaged (eg entries that do not
    ///  correspond to their MD5 token).
    ///
    ///  @return true if the logs were flushed okay, false if
    ///          there was a problem.
    //////////////////////////////////////////////////////////////
    
    function flushLogs() {

			trigger_error('Inside logging->flushLogs');

			// Start by cleaning the pending logfile
			$linestokeep = array();
			foreach (file($this->logpending) as $line) {
				// Filter out bad tokens
				$trans = new Transaction();
				$trans->fromString($line);
				if (!$trans->isValidToken($this->days)) continue;
				// If token passes filters, add to list to keep
				array_push($linestokeep, $line);
			}

			// And now we write all the good tokens back to
			// the pending log.
			if (is_writable($this->logpending)) {
				if (!$fp = fopen($this->logpending, 'w')) return false;
				if (!fwrite($fp, join("",$linestokeep))) return false;
				fclose($fp);
			} else return false;

			trigger_error('Leaving logging->flushLogs');
			return true;
    }


  }
?>