<?php

  //////////////////////////////////////////////////////////////
  ///
  ///  @class ProgressLock
  ///  @brief WebScriber ProgressLock class
  ///  @note  Copyright (c) 2005-2008 namesuppressed.
  ///
  ///  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
  ///  KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
  ///  WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
  ///  PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
  ///  OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
  ///  OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
  ///  OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
  ///  SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
  ///
  ///
  ///   Usage example:
  ///
  ///   @code
  ///  	// Block if we don't also have a unique ID set,
  ///		// create one, record it, and redirect.  This
  ///		// is to prevent the user hitting Refresh.
  ///	
  ///		$variable = 'uniq';
  ///		$url = $config->scripturl;
  ///		$logfile = $logging->logprogress;
  ///	
  ///		$temp = new ProgressLock();
  ///		if (!isset($_GET[$variable])) {
  ///			$savedok = $temp->preventRefresh($logfile, $url, $_POST, $variable);
  ///			if (!$savedok) {
  ///				show($templateError, $trans, 'Unable to open progress log');
  ///				trigger_error('Unable to open progress log');
  ///				trigger_error('--- end ---');
  ///				exit;
  ///			}
  ///			trigger_error("Redirecting to $url?$variable=" . $temp->uniqueID);
  ///			trigger_error('--- end ---');
  ///			exit;
  ///		}
  ///	
  ///	
  ///		// We're set to go.  Now we just decode the array
  ///		// and continue on our merry way.
  ///		$temp = new ProgressLock();
  ///		$okay = $temp->retrieveFromLog($logfile, $_GET[$variable]);
  ///		if (!$okay) {
  ///			// That token isn't in the log, so just go back to start.
  ///			header("Location: " . $url);
  ///			trigger_error('Redirecting to ' . $url);
  ///			trigger_error('--- end ---');
  ///			exit;
  ///		}
  ///		// We retrieved the data okay.
  ///		$_POST  = $temp->getData();
  ///		$action = $_POST['action'];
  ///   @endcode
  ///
  //////////////////////////////////////////////////////////////


class ProgressLock {



  var $uniqueID;  ///< A unique ID for this lock
  var $timestamp; ///< Timestamp
  var $processed; ///< Has data related to this lock been processed?
  var $data;      ///< The data to remember after we progress


  //////////////////////////////////////////////////////////////
  ///  Constructor for the ProgressLock class, creates an object
  ///  that stores HTML POST data which can be saved to a file,
  ///  somewhat like a .lock file.
  ///
  ///  @return   A ProgressLock object with a pregenerated
  ///            timestamp and with processed set to false.
  //////////////////////////////////////////////////////////////

  function ProgressLock() {
    list($usec, $sec) = explode(" ",microtime());
    $this->timestamp = (float)$usec + (float)$sec;
    $this->processed = "false";
  }



  //////////////////////////////////////////////////////////////
  ///  Sets the value of the processed variable.  Since the
  ///  processed variable doesn't actually take a boolean value
  ///  (it takes a string value), this function is required.
  ///
  ///  @param    var  a boolean value to assign to processed
  //////////////////////////////////////////////////////////////

  function setProcessed($var) {
    if ($var == true) $this->processed = "true";
    else $this->processed = "false";
  }

  
  
  //////////////////////////////////////////////////////////////
  ///  Returns the value of the processed variable as a boolean.
  ///
  ///  @return   the boolean value of the 'processed' variable
  //////////////////////////////////////////////////////////////

  function getProcessed() {
    if ($this->processed = "true") return true;
    if ($this->processed = "false") return false;
    return false;
  }



  //////////////////////////////////////////////////////////////
  ///  Checks whether this lock is still valid - that is, if the
  ///  timestamp on this lock still falls within the time period
  ///  according to $daysvalid before it expires.
  ///
  ///  @param    daysvalid  number of days lock is valid for
  ///  @return   true       if lock is still valid
  //////////////////////////////////////////////////////////////

  function isValid($daysvalid) {
    list($usec, $sec) = explode(" ",microtime());
    $currenttime = (float)$usec + (float)$sec;
    if ($this->timestamp < ($currenttime - $daysvalid*86400)) return false;
    if ($this->timestamp > $currenttime) return false;
    return true;
  }
  
  

  //////////////////////////////////////////////////////////////
  ///  Generates a unique identifier for this progress lock,
  ///  based on the current time at the instant this function is
  ///  called.  It is merely a way of generating a random token,
  ///  and the function uses this to set the uniqueID variable.
  ///
  ///  @return  the uniqueID that was just generated
  //////////////////////////////////////////////////////////////

  function generateUniqueID() {
    list($usec, $sec) = explode(" ",microtime());
    $seed = (float)$usec + (float)$sec;
    $string = "sAlT" . $seed;
    $this->uniqueID = md5($seed);
    return $this->uniqueID;
  }



  //////////////////////////////////////////////////////////////
  ///  Given any type of input (but most usually an HTTP POST
  ///  array), setData will set the data variable to the contents
  ///  of that input in a format that can be stored in a file.
  ///
  ///  @param    var  the data to be stored
  //////////////////////////////////////////////////////////////

  function setData($var) {
    $this->data = urlencode(serialize($var));
  }



  //////////////////////////////////////////////////////////////
  ///  Returns the contents of the data variable in its original,
  ///  non-serialized non-urlencoded format.
  ///
  ///  @return   the data in its original format
  //////////////////////////////////////////////////////////////

  function getData() {
    return unserialize(urldecode($this->data));
  }



	//////////////////////////////////////////////////////////////
	///  Store all the data stored in a variable/array in a file,
	///  then redirect the script.  This prevents users from
	///  accidentally triggering an action by hitting refresh.
	///
	///  @param  logfile   the file to save the data to
	///  @param  url       the url to redirect to
	///  @param  data      the data to recall after refreshing.
	///            You'll probably want this to be $_POST
	///  @param  variable  the name of the variable to put the
	///            unique ID in when redirecting
	///  @return true if data saved, false if not
	//////////////////////////////////////////////////////////////

	function preventRefresh($logfile, $url, $data, $variable) {
		$this->generateUniqueID();
		$this->setData($data);
		if (is_writable($logfile)) {
			// record unique ID and *all* details of transaction
			// into database or file or something.
			if (!$fp = fopen($logfile, 'a')) {
				return false;
			}
			fwrite($fp, $this->toString() . "\n");
			fclose($fp);
			if (!headers_sent()) {
	          header("Location: $url?$variable=" . $this->uniqueID);
	        }
	        else {
	          echo "<p>Unable to send redirection headers. ";
	          echo "If you see text similar to <code>#!/usr/bin/php</code> above, ";
	          echo "try deleting that line from the top of the email.php ";
	          echo "file in your Webscriber directory.  (The &lt;?php part should be ";
	          echo "at the very start of the file, with no spaces or blank lines ";
	          echo "before it.)</p>";
	        }
			return true;
		}
		else return false;
	}



	//////////////////////////////////////////////////////////////
	///  Retrieves data from a ProgressLock logfile.
	///
	///  @param  logfile  the file to save the data to
	///  @param  token  the token to find in the logfile
	///  @return true if token in logfile, false otherwise
	//////////////////////////////////////////////////////////////
	
	function retrieveFromLog($logfile, $token) {
		// Check if the token is in the progress log already.
		// If not, return to the default script screen.
		// If so, delete the uniq from the progress logs.
		// Get all lines beginning with token and store in $matches
		$matches = array();
		$remainder = array();
		$matchstring = "/^" . $token . "\|.*$/";
		foreach (file($logfile) as $line) {
			if (preg_match($matchstring, $line)) array_push($matches, $line);
			else array_push($remainder, $line);
		}
		if (sizeof($matches) < 1) return false;
		else {
			// Delete the unique ID now
			$fp = fopen($logfile, 'w');
			fwrite($fp, join("", $remainder));
			fclose($fp);
		}
		$this->fromString($matches[0]);
		return true;
	}



  //////////////////////////////////////////////////////////////
  ///  Takes a string and parses it, populating the variables
  ///  of this class according to the data.  To be used in
  ///  conjunction with the toString class, which creates the
  ///  strings in the appropriate format.
  ///
  ///  @param    str  the string to parse
  //////////////////////////////////////////////////////////////

  function fromString($str) {
    $elements = explode("|", trim($str));
    $this->uniqueID = $elements[0];
    $this->timestamp = $elements[1];
    $this->processed = $elements[2];
    $this->data = $elements[3];
  }



  //////////////////////////////////////////////////////////////
  ///  Constructs a string containing all the data from this
  ///  object, in such a format that it can be saved and
  ///  reparsed to load the data back into the object using the
  ///  fromString function.
  ///
  ///  @return   a string representing this object's data
  //////////////////////////////////////////////////////////////

  function toString() {
    $retval  = $this->uniqueID . '|' . $this->timestamp; 
    $retval .= '|' . $this->processed . '|' . $this->data;
    return $retval;
  }
  
  
  
	//////////////////////////////////////////////////////////////
	///  Removes entries from the ProgressLock logfile that are
	///  out of date or damage (eg entries that do not correspond
	///  to their MD5 token).
	///
	///  @param  logfile    full path to the logfile
	///  @param  daysvalid  number of days a lock is valid for
	///  @return true if the logs were flushed okay, false if
	///          there was a problem.
	//////////////////////////////////////////////////////////////
  
  function flushLogs($logfile, $daysvalid) {

		if (!is_readable($logfile)) return false;

		// Now we're going to start cleaning the progress log
		$linestokeep = array();
		foreach (file($logfile) as $line) {
			// Filter out bad tokens
			$temp = new ProgressLock();
			$temp->fromString($line);
			if (!$temp->isValid($daysvalid)) continue;
			// If token passes filters, add to list to keep
			array_push($linestokeep, $line);
		}
		if (is_writable($logfile)) {
			if (!$fp = fopen($logfile, 'w')) return false;
			if (!fwrite($fp, join("",$linestokeep))) return false;
			fclose($fp);
		} else return false;
  }

  
}
?>