<?php

  //////////////////////////////////////////////////////////////
  ///
  ///  @class Transaction
  ///  @brief WebScriber Transaction class
  ///  @note  Copyright (c) 2005-2008 namesuppressed.
  ///
  ///  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
  ///  KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
  ///  WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
  ///  PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
  ///  OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
  ///  OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
  ///  OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
  ///  SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
  ///
  //////////////////////////////////////////////////////////////


	require_once('action.php');
	require_once('visitor.php');


	class Transaction {

		var $token;    ///< MD5 of the other elements in the object
		var $subtoken; ///< MD5 of uniqid of other elements in object
		var $email;    ///< Email address to subscribe
		var $note;     ///< A note of how this address was obtained
		var $action;   ///< Which lists should we subscribe/unsub from?
		var $requestvisitor;  ///< Details of user initiating subscription
		var $confirmvisitor;  ///< Details of user confirming subscription



		//////////////////////////////////////////////////////////////
		///  Constructs a new Transaction object.
		///
		///  @return a Transaction object
		//////////////////////////////////////////////////////////////

		function Transaction() {
			$this->token = "No_token";
			$this->subtoken = "No_subtoken";
			$this->email = "No_email";
			$this->note = "No_note";
			$this->action = new Action();
			$this->requestvisitor = new Visitor();
			$this->confirmvisitor = new Visitor();
		}



		//////////////////////////////////////////////////////////////
		///  Gets the token associated with this Transaction
		///
		///  @return the token associated with this Transaction
		//////////////////////////////////////////////////////////////

		function getToken() {
			return $this->token;
		}



		//////////////////////////////////////////////////////////////
		///  Sets the token associated with this Transaction
		///
		///  @param  t  The new token for this object
		//////////////////////////////////////////////////////////////

		function setToken($t) {
			$this->token = $t;
		}



		//////////////////////////////////////////////////////////////
		///  Returns the email address associated with this
		///  Transaction.
		///
		///  @return  this transaction's email address
		//////////////////////////////////////////////////////////////

		function getEmail() {
			return $this->email;
		}


		//////////////////////////////////////////////////////////////
		///  Set the email address to subscribe/unsubscribe to all
		///  of these mailing lists in this transaction
		///
		///  @param e  the email address to un/subscribe
		//////////////////////////////////////////////////////////////

		function setEmail($e) {
			$this->email = strtolower($e);
		}



		//////////////////////////////////////////////////////////////
		///  Get the note associated with this Transaction
		///
		///  @return  The note associated with this Transaction
		//////////////////////////////////////////////////////////////

		function getNote() {
			return $this->note;
		}



		//////////////////////////////////////////////////////////////
		///  Set the note associated with this Transaction
		///
		///  @param n  The note to attach to this transaction
		//////////////////////////////////////////////////////////////

		function setNote($n) {
			$this->note = $n;
		}



		//////////////////////////////////////////////////////////////
		///  Set the visitor who requested this transaction to occur
		///
		///  @param n  A visitor object with details of who requested
		///            this transaction occur
		//////////////////////////////////////////////////////////////

		function setRequestVisitor($n) {
			$this->requestvisitor = $n;
		}



		//////////////////////////////////////////////////////////////
		///  Set the visitor who confirmed that the actions in this
		///  transaction should proceed
		///
		///  @param n  A Visitor object with details of who confirmed
		///            the transaction should proceed
		//////////////////////////////////////////////////////////////

		function setConfirmVisitor($n) {
			$this->confirmvisitor = $n;
		}



		//////////////////////////////////////////////////////////////
		///  Get details of the website visitor who requested this
		///  Transaction
		///
		///  @return a Visitor object with details of who requested
		///          the transaction to occur
		//////////////////////////////////////////////////////////////

		function getRequestVisitor() {
			return $this->requestvisitor;
		}



		//////////////////////////////////////////////////////////////
		///  Get details of the website visitor who confirmed this
		///  Transaction
		///
		///  @return a Visitor object with details of who confirmed
		///          the transaction
		//////////////////////////////////////////////////////////////

		function getConfirmVisitor() {
			return $this->confirmvisitor;
		}



		//////////////////////////////////////////////////////////////
		///  Add a list to subscribe to this Action
		///
		///  @param name  the name of the list to subscribe to
		//////////////////////////////////////////////////////////////

		function subscribeList($name) {
			$this->action->subscribeList($name);
			$this->generateToken();
		}



		//////////////////////////////////////////////////////////////
		///  Add a list to unsubscribe from to this Action
		///
		///  @param name  the name of the list to unsubscribe from
		//////////////////////////////////////////////////////////////

		function unsubscribeList($name) {
			$this->action->unsubscribeList($name);
			$this->generateToken();
		}



		//////////////////////////////////////////////////////////////
		///  Get the number of subscribe & unsubscribe tasks
		///  associated with this Action
		///
		///  @return the number of tasks associated with this Action
		//////////////////////////////////////////////////////////////

		function getNumTasks() {
			return $this->action->getNumTasks();
		}



		//////////////////////////////////////////////////////////////
		///  Get a particular task from the list of tasks associated
		///  with this Transaction
		///
		///  @param  val  the number of the task to get
		///  @return the task requested, or false if val is out of
		///          range
		//////////////////////////////////////////////////////////////

		function getTask($val) {
			return $this->action->getTask($val);
		}



		//////////////////////////////////////////////////////////////
		///  Returns the lists the user is subscribing to
		///
		///  @return An array of subscribe Actions
		//////////////////////////////////////////////////////////////

		function getToSubscribe() {
			return $this->action->getToSubscribe();
		}



		//////////////////////////////////////////////////////////////
		///  Returns the lists the user is unsubscribing from
		///
		///  @return An array of unsubscribe Actions
		//////////////////////////////////////////////////////////////

		function getToUnsubscribe() {
			return $this->action->getToUnsubscribe();
		}



		//////////////////////////////////////////////////////////////
		///  Set all the member variables of this object by parsing
		///  a string representation of the object.
		///
		///  @see   toString
		///  @param str  the string to parse for data
		//////////////////////////////////////////////////////////////

		function fromString($str) {
			$elements = explode("|", trim($str));
			$this->token = $elements[0];
			$this->subtoken = $elements[1];
			$this->email = $elements[2];
			$this->note = $elements[3];

			$temp = new Action();
			$temp->fromString($elements[4]);
			$this->action = $temp;

			$temp = new Visitor();
			$temp->fromString($elements[5]);
			$this->requestvisitor = $temp;

			$temp = new Visitor();
			$temp->fromString($elements[6]);
			$this->confirmvisitor = $temp;
		}



		//////////////////////////////////////////////////////////////
		///  Returns a string containing this Transaction's info
		///
		///  @return a string representation of this object
		//////////////////////////////////////////////////////////////

		function toString() {
			$retval  = "$this->token|$this->subtoken|$this->email|$this->note";
			$retval .= "|" . $this->action->toString();
			$retval .= "|" . $this->requestvisitor->toString();
			$retval .= "|" . $this->confirmvisitor->toString();
			return $retval;
		}



		//////////////////////////////////////////////////////////////
		///  Tests if the MD5 token associated with this transaction
		///  is valid or not.
		///
		///  @param  daysvalid  maximum number of days that a
		///            transaction is allowed to be valid for
		///  @return true if this transaction is valid within the
		///          given time period, false otherwise.
		//////////////////////////////////////////////////////////////

		function isValidToken($daysvalid) {
			// First check that the MD5 token is still valid.
			$seed  = $this->subtoken . $this->email . $this->note;
			$seed .= $this->action->toString();
			$seed .= $this->requestvisitor->toString();
			$generated = md5($seed);
			if ($this->token != $generated) return false;

			// Now we check that it falls within the correct time period.
			list($usec, $sec) = explode(" ",microtime());
			$currenttime = (float)$usec + (float)$sec;
			$timestamp = $this->requestvisitor->getTimestamp();
			if ($timestamp < ($currenttime - $daysvalid*86400)) return false;
			if ($timestamp > $currenttime) return false;

			// No problems, so return true - it's valid.
			return true;
		}



		//////////////////////////////////////////////////////////////
		///  Sets the Token to a value that is valid for the
		///  information contained in this Transaction object.
		//////////////////////////////////////////////////////////////

		function generateToken() {
			$subseed = $this->requestvisitor->toString();
			$this->subtoken = uniqid(md5(rand() . $subseed));
			$seed  = $this->subtoken . $this->email . $this->note;
			$seed .= $this->action->toString();
			$seed .= $this->requestvisitor->toString();
			$this->token = md5($seed);
		}



		//////////////////////////////////////////////////////////////
		///  Generates a timestamp for the visitor requesting this
		///  transaction to occur
		//////////////////////////////////////////////////////////////

		function generateRequestTimestamp() {
			$this->requestvisitor->generateTimestamp();
		}



		//////////////////////////////////////////////////////////////
		///  Generates a timestampe for the visitor confirming that
		///  this Transaction should proceed
		//////////////////////////////////////////////////////////////

		function generateConfirmTimestamp() {
			$this->confirmvisitor->generateTimestamp();
		}
	}
?>