<?php

  //////////////////////////////////////////////////////////////
  ///
  ///  @class Visitor
  ///  @brief WebScriber Visitor class
  ///  @note  Copyright (c) 2005-2008 namesuppressed.
  ///
  /// -----------------------------------------------------------
  ///
  ///  A visitor is someone who visits the mailing list manager
  ///  site in their web browser, eg to subscribe to a mailing
  ///  list or confirm a subscription.  A visitor has an IP
  ///  address, a web browser (also known as the User Agent), and
  ///  arrives at a particular time, the timestamp.  There might
  ///  also be details about the page the visitor came from, the
  ///  Referrer.
  ///
  ///  Why is the Visitor a separate class to the Transaction?
  ///  Because the visitor who subscribes is not necessarily the
  ///  same as the one who confirms the subscription.  They might
  ///  come from a different IP, use a different browser... in
  ///  fact, they might be a hacker trying to falsify a
  ///  subscription.  This is why we must record both the
  ///  "Request Visitor" (the one who makes the subscription
  ///  request) and the "Confirm Visitor" (the one who confirms
  ///  the request).
  ///
  /// -----------------------------------------------------------
  ///
  ///  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
  ///  KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
  ///  WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
  ///  PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
  ///  OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
  ///  OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
  ///  OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
  ///  SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
  ///
  //////////////////////////////////////////////////////////////



class Visitor {

  var $timestamp;   ///< Timestamp
  var $IP;          ///< IP Address
  var $useragent;   ///< UserAgent string
  var $referrer;    ///< Referring URL



	//////////////////////////////////////////////////////////////
	///  Constructor for the Visitor class
	///
	///  @return a Visitor object
	//////////////////////////////////////////////////////////////

  function Visitor() {
    $this->generateTimestamp();
    $this->IP = "No_IP";
    $this->useragent = "No useragent";
    $this->referrer = "No HTTP_REFERER";
  }



	//////////////////////////////////////////////////////////////
	///  Get the time that this visitor visited the website
	///
	///  @return a timestamp of when the visitor visited
	//////////////////////////////////////////////////////////////

  function getTimestamp() {
    return $this->timestamp;
  }


  
	//////////////////////////////////////////////////////////////
	///  Set the time that this visitor visited the website
	///
	///  @param  t  timestamp when this visitor visited the site
	//////////////////////////////////////////////////////////////

  function setTimestamp($t) {
    // Maybe do some pattern matching here
    // to ensure we are getting a timestamp?
    $this->timestamp = $t;
  }



	//////////////////////////////////////////////////////////////
	///  Generate a timestamp in this object for this exact
	///  second
	//////////////////////////////////////////////////////////////

  function generateTimestamp() {
    list($usec, $sec) = explode(" ",microtime());
    $this->timestamp = (float)$usec + (float)$sec;
  }



	//////////////////////////////////////////////////////////////
	///  Get the website that referred this visitor
	///
	///  @return  a string, the url of the referring website
	//////////////////////////////////////////////////////////////

  function getReferrer() {
    return $this->referrer;
  }


  
	//////////////////////////////////////////////////////////////
	///  Set the website that referred this visitor
	///
	///  @param  r  the url of the referring website
	//////////////////////////////////////////////////////////////

  function setReferrer($r) {
    $this->referrer = $r;
  }



	//////////////////////////////////////////////////////////////
	///  Get the Useragent (browser) that this visitor used
	///
	///  @return a string with the Useragent string
	//////////////////////////////////////////////////////////////

  function getUseragent() {
    return $this->useragent;
  }



	//////////////////////////////////////////////////////////////
	///  Set the Useragent (browser) used by this visitor
	///
	///  @param  ua  string with the brower Useragent string
	//////////////////////////////////////////////////////////////

  function setUseragent($ua) {
    $this->useragent = $ua;
  }


  
	//////////////////////////////////////////////////////////////
	///  Get the IP address of this visitor
	///
	///  @return a string with the visitor's IP address
	//////////////////////////////////////////////////////////////

  function getIP() {
    return $this->IP;
  }



	//////////////////////////////////////////////////////////////
	///  Set this visitor's IP address
	///
	///  @param ip  this user's IP address
	//////////////////////////////////////////////////////////////

  function setIP($ip) {
    $this->IP = $ip;
  }



	//////////////////////////////////////////////////////////////
	///  Set this visitor's IP address, useragent & referrer from
	///  the Server Environment variables
	//////////////////////////////////////////////////////////////

  function fromEnvironment() {
    $this->generateTimestamp();
  
    if (isset($_SERVER['REMOTE_ADDR']))
      $this->IP = $_SERVER['REMOTE_ADDR'];
    else $this->IP = "No_IP";

    if (isset($_SERVER['HTTP_USER_AGENT']))
      $this->useragent = $_SERVER['HTTP_USER_AGENT'];
    else $this->useragent = "No useragent";

    if (isset($_SERVER['HTTP_REFERER']))
      $this->referrer = $_SERVER['HTTP_REFERER'];
    else $this->referrer = "No HTTP_REFERER";
  }



	//////////////////////////////////////////////////////////////
	///  Set all the member variables of this object by parsing
	///  a string representation of the object.
	///
	///  @see   toString
	///  @param str  the string to parse for data
	//////////////////////////////////////////////////////////////

  function fromString($str) {
    $elements = explode(">", trim($str));
    $this->timestamp = $elements[0];
    $this->IP = $elements[1];
    $this->useragent = $elements[2];
    $this->referrer = $elements[3];
  }



  //////////////////////////////////////////////////////////////
	///  Returns a string containing this Visitor object's info
	///
	///  @see    fromString
	///  @return a string representation of this object
	//////////////////////////////////////////////////////////////

  function toString() {
    $retval  = "$this->timestamp>$this->IP>";
    $retval .= "$this->useragent>$this->referrer";
    return $retval;
  }
}