<?php

  //////////////////////////////////////////////////////////////
  ///
  ///  @class WSEmailer
  ///  @brief WebScriber WSEmailer class
  ///  @note  Copyright (c) 2005-2008 namesuppressed.
  ///
  ///  A wrapper for the Emailer class, providing WebScriber
  ///  specific functionality around the Emailer functions.
  ///
  ///  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
  ///  KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
  ///  WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
  ///  PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
  ///  OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
  ///  OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
  ///  OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
  ///  SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
  ///
  //////////////////////////////////////////////////////////////


  require_once('emailer.php');


  class WSEmailer {

    var $config;       ///< Reference to Configuration class
    var $listmanager;  ///< Reference to ListManager class
    var $emailer;      ///< Reference to Emailer class
    
    var $unlockcode;   ///< WebScriber unlocking code (duplicate)
    var $testingmode;  ///< Testing mode, disables sending email

    // Subject lines for each email message
    var $batchSubject;          ///< Subject line for Batch Subscription emails
    var $subscribeSubject;      ///< Subject line for Subscribe emails
    var $unsubscribeSubject;    ///< Subject line for Unsubscribe emails
    var $subConfirmSubject;     ///< Subject line for Subscription Confirmation emails
    var $unsubConfirmSubject;   ///< Subject line for Unsubcription Confirmation emails
    var $updateSubject;         ///< Subject line for Update emails
    var $updateConfirmSubject;  ///< Subject line for Update Confirmation emails

    // Email message template file locations
    var $emailBatchSub;         ///< Full path to Batch Subscription
    var $emailSubscribe;        ///< Full path to Subscribe template
    var $emailUnsubscribe;      ///< Full path to Unsubscribe template
    var $emailSubConfirmed;     ///< Full path to Subscribe Confirmed
    var $emailUnsubConfirmed;   ///< Full path to Unsubscribe Confirmed
    var $emailUpdate;           ///< Full path to Update template
    var $emailUpdateConfirmed;  ///< Full path to Update Confirmed template



    //////////////////////////////////////////////////////////////
    ///  Constructor for the WSEmailer class, handles the sending
    ///  of specific WebScriber email messages.
    ///
    ///  @param  config       a configuration object
    ///  @param  listmanager  reference to listmanager object
    ///  @return a WSEmailer object with the settings loaded from
    ///          the configuration file.
    //////////////////////////////////////////////////////////////

    function WSEmailer(&$config, &$listmanager) {
      $this->emailer     = new Emailer();
      $this->listmanager = &$listmanager;
      $this->config      = &$config;
      $this->unlockcode  = $config->unlockcode;
      $this->testingmode = $config->testingmode;
      $this->getSettingsFromConfig($config);
    }



    //////////////////////////////////////////////////////////////
    ///  Sets the locations of the email template files
    ///
    ///  @param basedir  Base directory where templates are stored
    //////////////////////////////////////////////////////////////

    function setEmailTemplates($basedir) {
      $this->emailBatchSub        = $basedir . 'email/batchsubscribe.txt';
      $this->emailSubscribe       = $basedir . 'email/subscribe.txt';
      $this->emailUnsubscribe     = $basedir . 'email/unsubscribe.txt';
      $this->emailSubConfirmed    = $basedir . 'email/subconfirmed.txt';
      $this->emailUnsubConfirmed  = $basedir . 'email/unsubconfirmed.txt';
      $this->emailUpdate          = $basedir . 'email/update.txt';
      $this->emailUpdateConfirmed = $basedir . 'email/updateconfirmed.txt';
    }
    
    
    
    //////////////////////////////////////////////////////////////
    ///  Sets the variables of this Emailer class depending on the
    ///  values in the configuration object
    ///
    ///  @param  config  A reference to a Configuration object
    ///  @return true if settings set correctly, false if
    ///          there was a problem setting values.
    //////////////////////////////////////////////////////////////

    function getSettingsFromConfig($config) {
    
      // Set the general email variables
      $this->setEmailTemplates($config->basedir);
      $this->emailer->setSenderName($config->sendername);
      $this->emailer->setSenderEmail($config->senderemail);
      $this->emailer->setUseBuiltInMail($config->builtinmail);
      $this->emailer->setEmailCopiesToSender($config->emailcopies);

      // Set up the SMTP Class
      $this->emailer->setSMTPHostname($config->smtphost);
      $this->emailer->setSMTPUser($config->smtpusername);
      $this->emailer->setSMTPPassword($config->smtppassword);
      $this->emailer->setSMTPRealm($config->smtprealm);
      $this->emailer->setSMTPDebug($config->smtpdebug);
      
      return true;
    }



    //////////////////////////////////////////////////////////////
    ///  Sets some variables in the Configuration depending
    ///
    ///  @param  config  A reference to a Configuration object
    ///  @return true if settings grabbed correctly, false if
    ///          there was a problem setting values.
    //////////////////////////////////////////////////////////////

    function setConfigurationSettings(&$config) {
      $config->sendername   = $this->emailer->getSenderName();
      $config->senderemail  = $this->emailer->getSenderEmail();
      $config->builtinmail  = $this->emailer->getUseBuiltInMail();
      $config->emailcopies  = $this->emailer->getEmailCopiesFromSender();
      $config->smtphost     = $this->emailer->getSMTPHostname();
      $config->smtpusername = $this->emailer->getSMTPUser();
      $config->smtppassword = $this->emailer->getSMTPPassword();
      $config->smtprealm    = $this->emailer->getSMTPRealm();
      $config->smtpdebug    = $this->emailer->getSMTPDebug();
      return true;
    }



    //////////////////////////////////////////////////////////////
    ///  Tests if the template files can be read
    ///
    ///  @return  true if the logfiles are readable and writable,
    ///           false otherwise
    //////////////////////////////////////////////////////////////

    function testFiles() {
      $emailtemplatesok  = is_readable($this->emailBatchSub);
      $emailtemplatesok &= is_readable($this->emailSubscribe);
      $emailtemplatesok &= is_readable($this->emailUnsubscribe);
      $emailtemplatesok &= is_readable($this->emailSubConfirmed);
      $emailtemplatesok &= is_readable($this->emailUnsubConfirmed);
      $emailtemplatesok &= is_readable($this->emailUpdate);
      $emailtemplatesok &= is_readable($this->emailUpdateConfirmed);
      return $emailtemplatesok;
    }
    
    
    
    //////////////////////////////////////////////////////////////
    /// Sends a Batch email to the address
    /// contained in the given transaction.
    ///
    /// @return true if the message was sent successfully,
    ///         false otherwise
    //////////////////////////////////////////////////////////////

    function sendBatchEmail($trans) {
      $subjectline   = $this->config->batchSubject;
      $emailtextfile = $this->emailBatchSub;
      return $this->sendEmail($trans, $subjectline, $emailtextfile, '');
    }




    //////////////////////////////////////////////////////////////
    /// Sends an Update email to the address
    /// contained in the given transaction.
    ///
    /// @return true if the message was sent successfully,
    ///         false otherwise
    //////////////////////////////////////////////////////////////

    function sendUpdateEmail($trans) {
      $subjectline   = $this->config->updateSubject;
      $emailtextfile = $this->emailUpdate;
      return $this->sendEmail($trans, $subjectline, $emailtextfile, '');
    }




    //////////////////////////////////////////////////////////////
    /// Sends an Update Confirmation email to the address
    /// contained in the given transaction.
    ///
    /// @return true if the message was sent successfully,
    ///         false otherwise
    //////////////////////////////////////////////////////////////

    function sendUpdateConfirmEmail($trans) {
      $subjectline   = $this->config->updateConfirmSubject;
      $emailtextfile = $this->emailUpdateConfirmed;
      return $this->sendEmail($trans, $subjectline, $emailtextfile, '');
    }




    //////////////////////////////////////////////////////////////
    /// Sends a subscription email to the address
    /// contained in the given transaction.
    ///
    /// @return true if the message was sent successfully,
    ///         false otherwise
    //////////////////////////////////////////////////////////////

    function sendSubscribeEmail($trans) {
      $subjectline   = $this->config->subscribeSubject;
      $emailtextfile = $this->emailSubscribe;
      return $this->sendEmail($trans, $subjectline, $emailtextfile, '');
    }




    //////////////////////////////////////////////////////////////
    /// Sends a subscription Confirmation email to the address
    /// contained in the given transaction.
    ///
    /// @return true if the message was sent successfully,
    ///         false otherwise
    //////////////////////////////////////////////////////////////

    function sendSubscribeConfirmEmail($trans) {
      $subjectline   = $this->config->subConfirmSubject;
      $emailtextfile = $this->emailSubConfirmed;
      return $this->sendEmail($trans, $subjectline, $emailtextfile, '');
    }




    //////////////////////////////////////////////////////////////
    /// Sends an unsubscription email to the address contained
    /// in the given transaction.
    ///
    /// @return true if the message was sent successfully,
    ///         false otherwise
    //////////////////////////////////////////////////////////////

    function sendUnsubscribeEmail($trans) {
      $subjectline   = $this->config->unsubscribeSubject;
      $emailtextfile = $this->emailUnsubscribe;
      return $this->sendEmail($trans, $subjectline, $emailtextfile, '');
    }




    //////////////////////////////////////////////////////////////
    /// Sends an unsubscription confirmation email to the address
    /// contained in the given transaction.
    ///
    /// @return true if the message was sent successfully,
    ///         false otherwise
    //////////////////////////////////////////////////////////////

    function sendUnsubConfirmEmail($trans) {
      $subjectline   = $this->config->unsubConfirmSubject;
      $emailtextfile = $this->emailUnsubConfirmed;
      return $this->sendEmail($trans, $subjectline, $emailtextfile, '');
    }



    //////////////////////////////////////////////////////////////
    /// Uses the Emailer class to send an email to the user,
    /// and adds an Unregistered warning if the unlocking
    /// code is invalid.
    ///
    /// @param  trans       the transaction to parse for data
    /// @param  subject     unparsed subject line of the email
    /// @param  body        the text of the email body (array)
    /// @param  header      unparsed extra email headers
    /// @return true if the message was sent successfully,
    ///         false otherwise
    //////////////////////////////////////////////////////////////

    function sendEmail($trans, $subject, $body, $header) {
      trigger_error("Inside WSEmailer->sendemail");

      // Now we send the email message, but bypass if just testing
      if (!$this->testingmode) {
      
        // Send a confirmation email to the user, with optional unreg message
        $emailtext  = $this->parse(file($body), $trans, '');
        $subj       = $this->parse($subject, $trans, '');
        $header     = $this->parse($header, $trans, '');

        // Add unregistered warning message
        if ($this->unlockcode != "avalidcode") {
          array_push($emailtext, "\n\n");
          array_push($emailtext, "----------------------------------------------------------------\n");
          array_push($emailtext, "This email was sent using an unregistered copy of namesuppressed\n");
          array_push($emailtext, "WebScriber: <http://www.namesuppressed.com/products/webscriber/>\n");
          array_push($emailtext, "\n\n");
        }

        // Send the actual email
        $okay = $this->emailer->sendEmail($trans->getEmail(), $subj, $emailtext, $header);
        if (!$okay) trigger_error('Can\'t send email!');
        trigger_error("Leaving WSEmailer->sendemail");
        return $okay;
      }   
      else {
        trigger_error("Testing Mode - no email sent");
      }

      trigger_error("Leaving WSEmailer->sendemail");
      return true;
    }




    //////////////////////////////////////////////////////////////
    ///  Parses a string (or an array of strings), replacing a set
    ///  of codes in the strings with their corresponding values.
    ///  The codes and values must be passed as an array.  The
    ///  codes is the array key, and what the code is to be
    ///  replaced with is the array result.  In the strings,
    ///  codes to be replaced must be surrounded with double
    ///  ampersands before and after them.  Here's an example:
    ///
    ///  @code
    ///    $document = '&&TEST&&';
    ///    $codes = array();
    ///    $codes['TEST'] = 'This was replaced by parse.';
    ///    parse($document, $codes);
    ///    echo $document;
    ///  @endcode
    ///
    ///  @param arraystring  an array of strings to parse
    ///  @param codes  an array of codes to be replaced
    ///  @return an array of parsed strings
    //////////////////////////////////////////////////////////////

    function parser($arraystring, $codes) {

      // Set up all the tags to be replaced
      $search = array();
      $replace = array();
      foreach ($codes as $key => $result) {
        $code = '/&&' . $key . '&&/i';
        array_push($search, $code);
        array_push($replace, $result);
      }

      // And now we actually perform the replace
      $arraystring = preg_replace($search, $replace, $arraystring);
      return $arraystring;
    }



    //////////////////////////////////////////////////////////////
    ///  Parses a string (or an array of strings), replacing a set
    ///  of predefined codes with their corresponding values.  The
    ///  values are drawn from the given transaction object,
    ///  listname, token and timevalue.
    ///
    ///  @param   arraystring  an array of strings to parse
    ///  @param   transaction  the current transaction object
    ///  @param   error  an error message to include in output
    ///  @return  an array of parsed strings
    //////////////////////////////////////////////////////////////

    function parse($arraystring, $transaction, $error) {

      trigger_error('Inside ezmlm->parse');

      if (!isset($arraystring)) trigger_error('Arraystring is null');
      if (!isset($transaction)) trigger_error('Transaction is null');

      $action   = '';
      $tasklist = '';
      $maillist = false;
      $numtasks = $transaction->getNumTasks();

      // Compute the replacement for &&TASKLIST&&
      if ($numtasks > 0) {
        for ($i = 0; $i < $numtasks; $i++) {
          $a = $transaction->getTask($i);
          $l = $this->listmanager->getList($a->listname);
          if ($l != false) {
            if ($i > 0) $tasklist .= "\n\n";
            $tasklist .= 'List Name: ';
            $tasklist .= $l->getName() . "\n";
            $tasklist .= 'Action: ';
            if ($a->action == 's') $tasklist .= 'Subscribe';
            if ($a->action == 'u') $tasklist .= 'Remove';
          }
          else trigger_error("No such list: $a->listname");
        }
      }

      // Decide what to put in the action field and
      // what to use as our maillist (if anything)
      if ($numtasks > 0) {
        if ($numtasks > 1) {
          $action = 'multiple';
          $maillist = false;
        }
        else {
          $a = $transaction->getTask(0);
          if ($a->action == 's') $action .= 'subscribe';
          if ($a->action == 'u') $action .= 'unsubscribe';
          $maillist = $this->listmanager->getList($a->listname);
        }
      }

      $codes = array();
      $codes['ACTION'] = $action;
      $codes['CONFIRMURL']  = $this->config->scripturl . "?t=" . $transaction->getToken();
      $codes['DATESTAMP']   = date("r", $transaction->requestvisitor->getTimestamp());
      $codes['EMAIL']       = $transaction->getEmail();
      $codes['ERROR']       = $error;
      $codes['EXPIRYDAYS']  = $this->config->days;
      $codes['IPADDRESS']   = $transaction->requestvisitor->getIP();
      if (isset($maillist) && $maillist != false) {
        $codes['LISTDESCRIPTION'] = $maillist->getDescription();
        $codes['LISTNAME']        = $maillist->getName();
        $codes['LISTSUBSCRIBE']   = $maillist->subscribe;
        $codes['LISTUNSUBSCRIBE'] = $maillist->unsubscribe;
      }
      $codes['NOTE']        = $transaction->getNote();
      $codes['SCRIPTURL']   = $this->config->scripturl;
      $codes['SENDEREMAIL'] = $this->config->senderemail;
      $codes['SENDERNAME']  = $this->config->sendername;
      $codes['TASKLIST']    = $tasklist;
      $codes['USERAGENT']   = $transaction->requestvisitor->getUseragent();

      $arrayresult = $this->parser($arraystring, $codes);
      trigger_error('Leaving ezmlm->parse');
      return $arrayresult;
    }

  }

?>