#!/usr/local/bin/php
<?php

//////////////////////////////////////////////////////////////
// Scroll down for credits, version and licence info
//////////////////////////////////////////////////////////////  

// Required global variables for debug function
$debuglogfile = '';
$debugmode = false;

// error handler function
function errorHandler ($errno, $errstr, $errfile, $errline) {
  global $debuglogfile;
  global $debugmode;
  $error   = date("Ymd H:i:s  ");
  $logfile = $debuglogfile;
  if ($debugmode == true) {
    switch ($errno) {
      case E_USER_ERROR:
        $error .= "FATAL [$errno] $errstr (line: $errline  file: $errfile)";
        $error .= "PHP ".PHP_VERSION." (".PHP_OS.")\n";
        error_log($error, 3, $logfile);
        echo "Unrecoverable error.  Check WebScriber debug log for details.";
        exit(1);
        break;
      case E_USER_WARNING:
        $error .= "WARNING [$errno] $errstr (line: $errline  file: $errfile)\n";
        error_log($error, 3, $logfile);
        break;
      case E_USER_NOTICE:
        $error .= "NOTICE [$errno] $errstr (line: $errline  file: $errfile)\n";
        error_log($error, 3, $logfile);
        break;
      default:
        $error .= "UNKNOWN [$errno] $errstr (line: $errline  file: $errfile)\n";
        error_log($error, 3, $logfile);
        break;
    }
  }
}

error_reporting(E_ERROR | E_WARNING | E_PARSE | E_NOTICE | E_ALL);
set_error_handler("errorHandler");



//////////////////////////////////////////////////////////////
///
///  @class   Webscriber
///  @brief   Main class - HTML display / CGI parser layer
///  @note    Copyright (c) 2008 namesuppressed.
///  @author  Kohan Ikin / namesuppressed
///  @author  support@namesuppressed.com
///
///  @version 1.18
///  @date    08 August 2008
///
///  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
///  KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
///  WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
///  PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
///  OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
///  OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
///  OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
///  SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
///
//////////////////////////////////////////////////////////////

class WebScriber {

  var $version = '1.18'; ///< Version of WebScriber
  
  var $configfile;       ///< Filename for configuration file
  var $config;           ///< Reference to Config object
  var $logging;          ///< Reference to Logging object
  var $ezmlm;            ///< Reference to Ezmlm object
  var $listmanager;      ///< Reference to ListManager object

  var $templateGeneric;       ///< Full path to templateGeneric file
  var $templateHeader;        ///< Full path to templateHeader file
  var $templateFooter;        ///< Full path to templateFooter file
  var $templateError;         ///< Full path to templateError file
  var $templateConfirmed;     ///< Full path to templateConfirmed file
  var $templateManage;        ///< Full path to templateManage file
  var $templateUpdated;       ///< Full path to templateUpdated file
  var $templateBatchRequest;  ///< Full path to templateBatchRequest file
  var $templateSingleRequest; ///< Full path to templateSingleRequest file

  var $classfiles = array('classes/smtp.php',          ///< List of class files
                      'classes/ezmlm.php',
                      'classes/action.php',
                      'classes/logging.php',
                      'classes/visitor.php',
                      'classes/emailer.php',
                      'classes/wsemailer.php',
                      'classes/maillist.php',
                      'classes/transaction.php',
                      'classes/progresslock.php',
                      'classes/configuration.php');


  //////////////////////////////////////////////////////////////
  ///  Constructor for the WebScriber class
  ///
  ///  @param fn Filename of the configuration file
  //////////////////////////////////////////////////////////////

  function WebScriber($fn) {
    $this->configfile = $fn;
  }



  //////////////////////////////////////////////////////////////
  ///  Check for presence of include files
  ///
  ///  @return true if all class files are readable
  ///          false otherwise
  //////////////////////////////////////////////////////////////

  function checkClassFiles() {
    $includefilesok = true;
    foreach($this->classfiles as $item) {
      $includefilesok &= is_readable($item);
    }
    return $includefilesok;
  }



  //////////////////////////////////////////////////////////////
  ///  Returns an array of class files that couldn't be opened
  ///
  ///  @return Array of strings of missing classes
  //////////////////////////////////////////////////////////////

  function getMissingClassFiles() {
    $missing = array();
    foreach ($this->classfiles as $item) {
      if (!is_readable($item)) array_push($missing, $item);
    }
    return $missing;
  }



  //////////////////////////////////////////////////////////////
  ///  Check for presence of HTML template files
  ///
  ///  @return true if all template files are readable
  ///          false otherwise
  //////////////////////////////////////////////////////////////

  function checkTemplateFiles() {
    $templatesok  = is_readable(realpath($this->templateGeneric));
    $templatesok &= is_readable(realpath($this->templateHeader));
    $templatesok &= is_readable(realpath($this->templateFooter));
    $templatesok &= is_readable(realpath($this->templateError));
    $templatesok &= is_readable(realpath($this->templateManage));
    $templatesok &= is_readable(realpath($this->templateUpdated));
    $templatesok &= is_readable(realpath($this->templateBatchRequest));
    $templatesok &= is_readable(realpath($this->templateSingleRequest));
    $templatesok &= is_readable(realpath($this->templateConfirmed));
    return $templatesok;
  }



  //////////////////////////////////////////////////////////////
  ///  Includes all the class files.  Only call this once you've
  ///  checked all the class files are there!
  ///
  ///  @see checkClassFiles
  //////////////////////////////////////////////////////////////

  function includeClassFiles() {
    foreach ($this->classfiles as $item) {
      require_once($item);
    }
  }



  //////////////////////////////////////////////////////////////
  ///  Shows the contents of a textfile, parsing it and
  ///  optionally displaying an error message as well.
  //////////////////////////////////////////////////////////////

  function show($filename, $transaction, $error) {
    trigger_error("Inside show: $error");
    $output  = $this->ezmlm->parse(implode("", file($this->templateHeader)), $transaction, $error);
    $output .= $this->ezmlm->parse(implode("", file($filename)), $transaction, $error);
    if ($this->config->unlockcode != "avalidcode") {
      $output .= "<p><b>This is an evaluation copy of ";
      $output .= "<a href=\"http://www.namesuppressed.com/products/webscriber/\">";
      $output .= "namesuppressed Webscriber</a>!  Purchasing the script will remove ";
      $output .= "this message.</b></p>";
    }
    $output .= $this->ezmlm->parse(implode("", file($this->templateFooter)), $transaction, $error);
    trigger_error("Leaving show");
    echo $output;
  }



  //////////////////////////////////////////////////////////////
  ///  Description
  //////////////////////////////////////////////////////////////

  function accessViaToken($token) {

    // The user has called the script with a token, presumably as part
    // of a URL they found in an email we sent them.  So let's use it!
    trigger_error('Token - user has accessed script using a token.');

    // Create a visitor....
    $visitor = new Visitor();
    $visitor->fromEnvironment();

    // Can we confirm this token?  If not, show the reason.
    $errorcode = $this->ezmlm->confirm($token, $visitor);
    if ($errorcode != ERR_OK) {
      trigger_error('errorcode is NOT ok');
      $errormsg = $this->ezmlm->errorToString($errorcode);
      if (strlen($errormsg) == 0) $errormsg = 'Error while confirming, errorcode: ' . $errorcode;
      $temp = new Transaction();
      $this->show($this->templateError, $temp, $errormsg);
      trigger_error("errorcode: $errorcode  message: $errormsg");
      return;
    }

    // Let's get details about the entry we just confirmed.
    $trans2 = $this->logging->getConfirmed($token);
    if ($trans2 == false) {
      $temp = new Transaction();
      $this->show($this->templateError, $temp, "It claims bad things.  No confirmation matches.");
      return;
    }

    // All is well with the world.
    $this->show($this->templateConfirmed, $trans2, '');
  }



  //////////////////////////////////////////////////////////////
  ///  Shows an error message about the logfiles
  //////////////////////////////////////////////////////////////

  function showLogfileError() {
    // We're having problems loading the log files, so we really
    // need to warn the user of this so they can correct the
    // problems in their configuration file and installation.
    echo "<br />";
    echo "There was a problem accessing the logfiles required by this ";
    echo "script.<br /><br />Please check your config.txt file and ensure the ";
    echo "BASEDIRECTORY variable has been set correctly, that the log ";
    echo "files exist, and that they are all readable and <i><b>writable</b></i>.  If you ";
    echo "are unsure what is wrong, the readme file may be able to help you.";

    $tmpfoldername = $this->config->basedir . 'logs';
    if (!is_readable($tmpfoldername) || !is_writable($tmpfoldername)) {
        echo "<br /><br /><b>Hint:</b> It looks like the file permissions on the ";
        echo "<code>" . $tmpfoldername . "</code> directory itself are incorrect. ";
        echo "Even if the files appear to be in the right folder and have the ";
        echo "correct permissions, the file permissions on the folder/directory ";
        echo "may prevent everything inside the folder from being readable. ";
        echo "Please check that the folder has read and write permissions for the ";
        echo "owner and group the file is assigned to.  The group should be the ";
        echo "'www' group.  On Unix systems, the permissions should be 'mode 664'. ";
        echo "On Mac systems, use Get Info (command-I) on the ";
        echo "<code>" . $tmpfoldername . "</code> ";
        echo "folder and ensure that the Ownership & Permissions access setting ";
        echo "for Others is set to 'Read & Write'.<br /><br />";
    } 

    echo "<pre>";
    echo "Logpending:   " . $this->logging->logpending . "<br />";
    echo "Logprogress:  " . $this->logging->logprogress . "<br />";
    echo "Logconfirmed: " . $this->logging->logconfirmed . "<br />";
    echo "Logdebug:     " . $this->logging->logdebug . "<br /><br/>";
    echo "</pre>";
    
    $missing = $this->logging->getMissingLogFiles();
    if (count($missing) > 0) {
      echo "The following log files couldn't be loaded:<br />";
      echo "<pre>";
      foreach ($missing as $item) echo "$item<br />";
      echo "</pre>";
    }

    // Let's try to be extra helpful - check if the last character
    // of the basedirectory is a forward slash, this is necessary.
    $lastchar = $this->config->basedir[strlen($this->config->basedir) - 1];
    if ($lastchar != '/' && $lastchar != '\\') {
      echo "<b>Hint:</b> It looks like you forgot the trailing slash ";
      echo "at the end of the BASEDIRECTORY variable.<br />";
    }
    if ($lastchar == ' ') {
      echo "<b>Hint:</b> It looks like there's an extra space at the ";
      echo "end of the BASEDIRECTORY variable.  Please delete it, and ";
      echo "ensure it ends in a trailing slash.<br />";
    }
  }



  //////////////////////////////////////////////////////////////
  ///  Shows an error message about the template files
  //////////////////////////////////////////////////////////////

  function showTemplateError() {
    // We're having problems loading some of the template files,
    // so we need to warn the user of this so they can correct
    // the problems in their installation.
    echo "<br />";
    echo "There was a problem loading the HTML template files ";
    echo "required by this script.<br />Please check that the ";
    echo "BASEDIRECTORY variable in your config.txt configuration file ";
    echo "is correct, and that each of the HTML template files exists ";
    echo "and is readable.  If you are unsure what is wrong, the readme ";
    echo "file may be able to help you.";
    
    $tmpfoldername = $this->config->basedir . 'html';
    if (!is_readable($tmpfoldername)) {
        echo "<br /><br /><b>Hint:</b> It looks like the file permissions on the ";
        echo "<code>" . $tmpfoldername . "</code> directory itself are incorrect. ";
        echo "Even if the files appear to be in the right folder and have the ";
        echo "correct permissions, the file permissions on the folder/directory ";
        echo "may prevent everything inside the folder from being readable. ";
        echo "Please check that the folder has read permissions for everyone. ";
        echo "On Unix systems, this is called 'mode 644'.  On Mac systems, ";
        echo "use Get Info (command-I) on the <code>" . $tmpfoldername . "</code> ";
        echo "folder and ensure that the Ownership & Permissions access setting ";
        echo "for Others is set to 'Read only'.<br /><br />";
    }
  }



  //////////////////////////////////////////////////////////////
  ///  Sets the locations of the HTML template files
  ///
  ///  @param bd  Base directory where the HTML template files
  ///             are located
  //////////////////////////////////////////////////////////////

  function setTemplateLocations($bd) {
    $this->templateGeneric       = $bd . 'html/generic.html';
    $this->templateHeader        = $bd . 'html/header.html';
    $this->templateFooter        = $bd . 'html/footer.html';
    $this->templateError         = $bd . 'html/error.html';
    $this->templateManage        = $bd . 'html/manage.html';
    $this->templateUpdated       = $bd . 'html/updated.html';
    $this->templateBatchRequest  = $bd . 'html/batchrequest.html';
    $this->templateSingleRequest = $bd . 'html/singlerequest.html';
    $this->templateConfirmed     = $bd . 'html/confirmed.html';
  }




  //////////////////////////////////////////////////////////////
  ///  Sets up the debug logging system in this class.
  ///
  ///  @return true if the debug system can be setup, false if
  ///          there was a problem setting it up.
  //////////////////////////////////////////////////////////////

  function setupDebugSystem() {
    
    if (isset($this->logging) && isset($this->config)) {
      global $debuglogfile;
      global $debugmode;
      $debuglogfile = $this->logging->logdebug;
      $debugmode    = $this->config->debugmode;
      return true;
    }
    
    $error = 'Problem setting up debug system';
    $this->show($this->templateError, $trans, $error);
    return false;
  }



  //////////////////////////////////////////////////////////////
  ///  Checks for problems in the configuration settings
  ///
  ///  @return true if the configuration is okay, false otherwise
  //////////////////////////////////////////////////////////////

  function checkConfiguration() {

    if (!$this->checkClassFiles()) {
      echo "<h2>There were script errors:</h2>";
      echo "<br />";
      echo "The script was unable to load the files from the classes ";
      echo "directory.<br />Please ensure you have uploaded the 'classes' ";
      echo "directory, and that the read/write permissions have been set ";
      echo "appropriately.  If you are unsure what is wrong, the readme ";
      echo "file may be able to help you.";
      echo "<br /><br />";
      
      $tmpfoldername = 'classes';
	  if (!is_readable($tmpfoldername)) {
		echo "<b>Hint:</b> It looks like the file permissions on the ";
		echo "<code>" . $tmpfoldername . "</code> directory itself are incorrect. ";
		echo "Even if the files appear to be in the right folder and have the ";
		echo "correct permissions, the file permissions on the folder/directory ";
		echo "may prevent everything inside the folder from being readable. ";
		echo "Please check that the folder has read permissions for everyone. ";
		echo "On Unix systems, this is called 'mode 644'.  On Mac systems, ";
		echo "use Get Info (command-I) on the <code>" . $tmpfoldername . "</code> ";
		echo "folder and ensure that the Ownership & Permissions access setting ";
		echo "for Others is set to 'Read only'.<br /><br />";
	  }
    
      echo "The following class files couldn't be loaded:<br />";
      echo "<pre>";
      $missing = $this->getMissingClassFiles();
      foreach ($missing as $item) echo "$item - (Full path: " . getcwd() . "/$item)<br />";
      echo "</pre>";
      exit;
    }
    else $this->includeClassFiles();


    // First we need to verify our inputs and do some integrity checking
    // on the script.  Can we access our configuration file?
    $this->config  = new Configuration();
    $configfileok = $this->config->load($this->configfile);
    if (!$configfileok) {
      // We're having problems loading the log files, so we really
      // need to warn the user of this so they can correct the
      // problems in their configuration file and installation.
      echo "<br />";
      echo "The script was unable to load your configuration file.<br />";
      echo "Please ensure that config.txt is stored in the same directory ";
      echo "as your email.php file, and that the read/write permissions ";
      echo "have been set appropriately. If you are unsure what is wrong, ";
      echo "the readme file may be able to help you.";
      return false;
    }
    
    if (!$this->config->testConfigurationOkay()) {
      $temperrors = $this->config->getConfigurationErrors();
      echo '<h2>There were script errors:</h2><br />';
      foreach ($temperrors as $hint) {
        echo '<br />';
        echo '<b>Hint:</b> ' . $hint . '<br />';
      }
      return false;
    }

    // Kewl, that all works out fine... so now we can set the locations of
    // our template files, and check to see that they are all readable too!
    $this->setTemplateLocations($this->config->basedir);
    if (!$this->checkTemplateFiles()) {
      $this->showTemplateError();
      return false;
    }

    // The configuration is okay, so we can return true now.
    return true;
  }





  //////////////////////////////////////////////////////////////
  /// Display Generic Page
  ///
  /// @param trans  The transaction to parse when displaying
  ///               the generic page
  //////////////////////////////////////////////////////////////

  function displayGenericPage($trans) {

    // We don't have any parameters to work with, since the user didn't
    // submit any with their request.  So now we just display the
    // Generic template in the browser, and let them enter their email
    // details if they want to, and go from there.

    $output = $this->ezmlm->parse(implode("", file($this->templateGeneric)), $trans, '');
    if (strlen($output) < 1) {
      // If the template is empty, or for some reason we're not
      // outputting anything to the browser, we need to display
      // some kind of error message to the user.
      echo "<br />";
      echo "There was an error displaying the Generic template. <br />";
      echo "Please check the \$basedir variable in your configuration.";
      echo "<br />If you are using a Windows version of PHP, please ";
      echo "ensure your \$basedir variable begins with the drive name, ";
      echo "eg 'c:'";
      trigger_error('Error displaying Generic template.');
    }
    else $this->show($this->templateGeneric, $trans, '');
  }



  //////////////////////////////////////////////////////////////
  ///  Prevents an accidental resubmission of form data if the
  ///  user hits the refresh button on their browser.
  ///
  ///  @param v   the name of the URL to pass the unique token on
  ///  @param url the url of this script, so we can refresh to it
  ///  @param logfile  full path to a logfile to store data in
  //////////////////////////////////////////////////////////////

  function preventAccidentalRefresh($v, $url2, $log) {

    // Block if we don't also have a unique ID set,
    // create one, record it, and redirect.  This
    // is to prevent the user hitting Refresh.

    $variable  = $v;
    $url       = $url2;
    $logfile   = $log;
    $daysvalid = 1;

    $temp = new ProgressLock();
    $temp->flushLogs($logfile, $daysvalid);

    // If the unique variable isn't set, there's a
    // risk of the user hitting refresh and executing
    // the same request again.  So we preventRefresh,
    // and redirect to continue execution.

    if (!isset($_GET[$variable])) {
      $savedok = $temp->preventRefresh($logfile, $url, $_POST, $variable);
      if (!$savedok) {
        $this->show($this->templateError, $trans, 'Unable to open progress log');
        trigger_error('Unable to open progress log');
        return false;
      }
      // The redirection occurs in preventRefresh.
      // Progresslock.php would be a better location
      // to trigger the error, but we're trying to
      // keep the trigger_errors out of other classes.
      trigger_error("Redirecting to $url?$variable=" . $temp->uniqueID);
      return false;
    }


    // If the unique variable *is* set, which it clearly
    // must be if we reach here, then we just need to
    // reload the data into the POST variable, decode
    // the array and continue on our merry way.

    $temp = new ProgressLock();
    $okay = $temp->retrieveFromLog($logfile, $_GET[$variable]);
    if (!$okay) {
      // That token isn't in the log, so just go back to start.
      
	  if (!headers_sent()) {
	    header("Location: " . $url);
        trigger_error('Redirecting to ' . $url);
      }
      else {
        echo "<p>Unable to send redirection headers. ";
        echo "If text similar to #!/usr/bin/php appears above, ";
        echo "try deleting that line from the top of the email.php file.</p>";
        trigger_error('Error with redirection headers - possible shebang bug?');
      }

      return false;
    }

    // It's okay, so restore the POST data and return true
    $_POST = $temp->getData();
    return true;
  }



  //////////////////////////////////////////////////////////////
  ///  setupLoggingSystem
  //////////////////////////////////////////////////////////////

  function setupLoggingSystem() {
    // Test that the logfiles are okay
    $this->logging = new Logging($this->config->basedir, $this->config->days);
    $logfilesok = $this->logging->testFiles();
    if (!$logfilesok) {
      $this->showLogfileError();
      trigger_error('Problem when testing logfiles');
      return false;
    }
    $this->logging->flushLogs();
    return true;
  }



  //////////////////////////////////////////////////////////////
  /// Manage
  ///
  /// @param trans The transaction you want to manage
  //////////////////////////////////////////////////////////////

  function manage($trans) {

    $alllists = $this->listmanager->getAllLists();

    // Construct the subscription management form that should be
    // included on the subscription manager page listing all
    // the users subscriptions and what they can subscribe to.
    $formhtml  = "<form action=\"" . $this->config->scripturl;
    $formhtml .= "\" method=\"post\">\n";
    foreach ($alllists as $list) {
      $isSubscribed = $this->listmanager->isSubscribed($trans->getEmail(),
                                           $list->getID());
      if ($isSubscribed == true || !$list->isPrivate()) {
        $formhtml .= "<p><input type=\"checkbox\" name=\"lists[";
        $formhtml .= $list->getID() . "]\" ";
        if ($isSubscribed) $formhtml .= "CHECKED";
        $formhtml .= " /><b>" . $list->getName() . "</b><br />\n";
        $formhtml .= $list->getDescription() . "<br/>";
        if ($list->isPrivate()) {
          $formhtml .= "<i>This is a private list.  If you ";
          $formhtml .= "unsubscribe, you will not be able to ";
          $formhtml .= "resubscribe.</i></p>";
        }
        else $formhtml .= "</p>";
        $formhtml .=  "\n";
      }
    }
    $formhtml .= "  <input type=\"hidden\" name=\"email\" value=\"";
    $formhtml .= $trans->getEmail() . "\" />";
    $formhtml .= "  <input type=\"hidden\" name=\"action\" ";
    $formhtml .= "value=\"update\" />";
    $formhtml .= "  <p align=\"center\"><input type=\"submit\" ";
    $formhtml .= "value=\"Update Subscriptions\" /></p>\n";
    $formhtml .= "</form>";

    // Interpolate the $templateManage page to include the subscriptions
    // form and output it to the user's browser.
    $search  = array('/&&SUBSCRIPTIONSFORM&&/i');
    $replace = array($formhtml);
    $output  = preg_replace($search, $replace, file($this->templateManage));
    $header  = $this->ezmlm->parse(implode("", file($this->templateHeader)), $trans, '');
    $middle  = $this->ezmlm->parse(implode("", $output), $trans, '');
    $footer  = $this->ezmlm->parse(implode("", file($this->templateFooter)), $trans, '');
    echo $header . $middle . $footer;
  }



  //////////////////////////////////////////////////////////////
  /// Update
  ///
  /// @param trans The transaction you want to update
  //////////////////////////////////////////////////////////////

  function update($trans) {
    trigger_error('Inside Webscriber->update');
  
    // Grab the lists the visitor wants to subscribe to
    if (isset($_POST['lists'])) $listarray = $_POST['lists'];
    else $listarray = array();

    // Construct the transaction object
    $alllists = $this->listmanager->getAllLists();
    foreach ($alllists as $list) {
      $tempid    = $list->getID();
      $tempemail = $trans->getEmail();
      if (isset($listarray[$tempid]) && !$this->listmanager->isSubscribed($tempemail, $tempid)) {
        $trans->subscribeList($tempid);
      }
      else {
        if (!isset($listarray[$tempid]) && $this->listmanager->isSubscribed($tempemail, $tempid))
          $trans->unsubscribeList($tempid);
      }
    }

    // Now update the subscriptions and output to browser
    if ($trans->getNumTasks() > 0) {
      $errorcode = $this->ezmlm->update($trans);
      if ($errorcode == ERR_OK) {

        // If we're only unsubscribing and we're using single
        // opt-out, then show the confirmation template page.
        if ($this->config->singleoptout &&
          (count($trans->getToSubscribe()) == 0)) {
          $this->show($this->templateConfirmed, $trans, '');
        }
        else {
          if ($trans->getNumTasks() == 1) $this->show($this->templateSingleRequest, $trans, '');
          else $this->show($this->templateUpdated, $trans, '');
        }
      }
      else {
        $errormsg = $this->ezmlm->errorToString($errorcode);
        if (strlen($errormsg) == 0) {
          $errormsg  = 'There was an error updating your subscriptions. ';
          $errormsg .= 'Code: ' . $errorcode;
        }
        $this->show($this->templateError, $trans, $errormsg);
      }
    }
    else $this->show($this->templateError, $trans, "You didn't make any changes to your subscriptions.");
    trigger_error('Leaving Webscriber->update');
  }





  //////////////////////////////////////////////////////////////
  /// This function is used to add multiple email addresses at
  /// once.  It takes the list of emails from $_POST['email']
  ///
  /// @param trans The transaction you want to update
  //////////////////////////////////////////////////////////////

  function multi($trans) {
    if (isset($_POST['pw']) && $_POST['pw'] != $this->config->password) {
      $this->show($this->templateError, $trans, "Bad password");
      return;
    }
    if (!isset($_POST['note']) || empty($_POST['note'])) {
      $this->show($this->templateError, $trans, "You must provide a note of how you obtained the addresses.");
      return;
    }
    if (strtolower($_POST['note']) == 'no_note') {
      $this->show($this->templateError, $trans, "Sorry, unacceptable note.  Please be more specific.");
      return;
    }
    $emailarray = split("\n", $_POST['email']);
    if (count($emailarray) > ($this->config->maxemails/2)) {
      $errormessage  = "There is a maximum of ". $this->config->maxemails/2;
      $errormessage .= " emails that can be subscribed at one time. ";
      $errormessage .= "No addresses have been subscribed.";
      $this->show($this->templateError, $trans, $errormessage);
      return;
    }
    $goodaddies = array();
    $subscribedaddies = array();
    $cantemailaddies = array();
    $badaddies = array();
    for ($i = 0; $i < count($emailarray); $i++) {
      $emailarray[$i] = $this->listmanager->washAddress($emailarray[$i]);
      if ($this->listmanager->isValidEmail($emailarray[$i])) {
        $trans2 = new Transaction();
        $visitor2 = new Visitor();
        $visitor2->fromEnvironment();
        $trans2->setRequestVisitor($visitor2);
        $trans2->setEmail($emailarray[$i]);
        if (isset($_POST['note']) && strlen($_POST['note']) > 0)
          $trans2->setNote($_POST['note']);
        $errorcode = $this->ezmlm->batch($trans2, $_POST['list']);
        if ($errorcode == ERR_NO_SUCH_LIST) {
          $this->show($this->templateError, $trans, "There is no list with that ID.");
          return;
        }
        switch ($errorcode) {
          case ERR_OK:
            array_push($goodaddies, $emailarray[$i]);
            break;
          case ERR_ALREADY_SUBSCRIBED:
            array_push($subscribedaddies, $emailarray[$i]);
            break;
          case ERR_CANT_EMAIL:
            array_push($cantemailaddies, $emailarray[$i]);
            break;
          default:
            array_push($badaddies, $emailarray[$i]);
            break;
        }
      }
      else {
        // If the string is not an empty string, display
        // it in the list of problem addresses
        if (strlen($emailarray[$i]) > 0)
          array_push($badaddies, $emailarray[$i]);
      }
    }
    if (count($badaddies) == 0 &&
        count($subscribedaddies) == 0 &&
        count($cantemailaddies) == 0) {
      $this->show($this->templateBatchRequest, $trans, '');
    }
    else {
      $errormessage  = "There were problems with your batch subscriptions.<br />";

      if (count($subscribedaddies) > 0) {
        $errormessage .= "<br />These addresses were already subscribed:<br/>";
        for ($i=0; $i < count($subscribedaddies); $i++) {
          $errormessage .= "$subscribedaddies[$i] <br />";
        }
      }

      if (count($cantemailaddies) > 0) {
        $errormessage .= "<br />There were problems sending email to these addresses:<br/>";
        for ($i=0; $i < count($cantemailaddies); $i++) {
          $errormessage .= "$cantemailaddies[$i] <br />";
        }
      }

      if (count($badaddies) > 0) {
        $errormessage .= "<br />These addresses were invalid or had other problems:<br/>";
        for ($i=0; $i < count($badaddies); $i++) {
          $errormessage .= "$badaddies[$i] <br />";
        }
      }

      $errormessage .= "<br />" . count($goodaddies) . " other addresses were emailed successfully.";
      $this->show($this->templateError, $trans, $errormessage);
    }
  }






  //////////////////////////////////////////////////////////////
  ///  Subscribe / Unsubscribe
  ///
  ///  @param trans    The transaction to subscribe
  ///  @param isUnsub  set to true if unsubscribing
  //////////////////////////////////////////////////////////////

  function subscribeUnsubscribe($trans, $isUnsub) {
    
    trigger_error('Inside Webscriber->subscribeUnsubscribe');
  
    if (isset($_POST['pw']) && $_POST['pw'] == $this->config->password) {
      if (isset($_POST['note'])) $trans->setNote($_POST['note']);
    }
    $error = '';

    // If we're subscribing, do this
    if (!$isUnsub) {
      $error = 'Error while subscribing';
      $trans->subscribeList($_POST['list']);
      $errorcode = $this->ezmlm->subscribe($trans, $_POST['list']);
    }

    // But if we're unsubscribing, do this
    if ($isUnsub == true) {
      $error = 'Error while unsubscribing';
      $trans->unsubscribeList($_POST['list']);
      $errorcode = $this->ezmlm->unsubscribe($trans, $_POST['list']);
    }

    if ($errorcode == ERR_OK) {
      // Edit to fix bug discovered by Kathy
      if ($isUnsub && $this->config->singleoptout)
        $this->show($this->templateConfirmed, $trans, '');
      else
        $this->show($this->templateSingleRequest, $trans, '');
    }
    else {
      $errormsg = $this->ezmlm->errorToString($errorcode);
      if (strlen($errormsg) == 0) $errormsg = $error;
      $this->show($this->templateError, $trans, $errormsg);
    }
    
    trigger_error('Leaving Webscriber->subscribeUnsubscribe');
  }





  //////////////////////////////////////////////////////////////
  ///  Run the WebScriber script
  //////////////////////////////////////////////////////////////

  function run() {
  
    global $debugmode;  // Don't remove this line!!
  
    if (!$this->checkConfiguration()) exit;
    if (!$this->setupLoggingSystem()) exit;
    if (!$this->setupDebugSystem())   exit;

    // Record in the logs that we're starting the script
    trigger_error('--- begin ---');
    trigger_error('WebScriber v' . $this->version);
    trigger_error("Running on PHP ".PHP_VERSION." (".PHP_OS.")");

    // Setup the ezmlm manager and the ListManager
    $this->ezmlm       = new Ezmlm($this->config);
    $this->listmanager = new ListManager($this->config->path, $this->config->listbase);
    $this->listmanager->setAllLists($this->config->lists);

    // Check if the user is trying to submit a confirmation token.  This will
    // be via a url like example.com/cgi-bin/email.php?t=blahblahblah.  We
    // use "t" as our token parameter to keep the URL as short as possible.
    if (isset($_GET['t'])) {
      $this->accessViaToken($_GET['t']);
      trigger_error('--- end ---');
      exit;
    }


    // Nearly everything else needs us to create a 
    // transaction and a visitor....
    $trans   = new Transaction();
    $visitor = new Visitor();
    $visitor->fromEnvironment();
    $trans->setRequestVisitor($visitor);

    // No token was submitted, so maybe the user is trying to execute an
    // action?  (eg Such as managing their subscriptions, subscribing to
    // a list and so on.)
    if (isset($_POST['action']) || isset($_GET['uniq'])) {

      // Prevent the script from repeating the same action
      // if they accidentally click the Refresh button
      if (!$this->preventAccidentalRefresh('uniq', $this->config->scripturl, $this->logging->logprogress)) {
        trigger_error('--- end ---');
        exit;
      }

      // We retrieved the data okay.
      $action = $_POST['action'];

      // First we must deal with the case where $_POST['email'] may
      // contain multiple email addresses (ie when the administrator
      // is trying to subscribe multiple email addresses at once).
      if ($action=='multi') {
        $this->multi($trans);
        trigger_error('--- end ---');
        exit;
      }

      // All other actions from this point in time regard $_POST['email']
      // as containing just a single address, so we can sanitize it in
      // the following manner.  We check that the email variable is a
      // correct email address, and if it's not we abort with an error.
      if (isset($_POST['email'])) {
        if(!$this->listmanager->isValidEmail($_POST['email'])) {
          // Invalid email address, return an error
          $this->show($this->templateError, $trans, "Invalid email address.");
          trigger_error('--- end ---');
          exit;
        }
        $trans->setEmail($this->listmanager->washAddress($_POST['email']));
      }

      // Perform the update action first, as it doesn't need the
      // $list variable defined.  It doesn't need $lists defined
      // either - if the customer unsubscribes all lists, it
      // will be empty.
      if ($action == 'update') {
        $this->update($trans);
        trigger_error('--- end ---');
        exit;
      }

      // Maybe the user is trying to manage their subscriptions?
      // Check for the 'manage' action and an email address, then
      // act appropriately if they're found.
      if ($action == 'manage' && isset($_POST['email'])) {
        $this->manage($trans);
        trigger_error('--- end ---');
        exit;
      }

      // Looks like the user is trying to subscribe to / unsubscribe from a
      // list.  Assuming we have a list and an email defined, we're set to
      // go and process their request - so let's do it!
      if (!isset($_POST['list']) || !isset($_POST['email'])) {
        $this->show($this->templateError, $trans, 'Missing list or email address');
        trigger_error('--- end ---');
        exit;
      }

      // Okay, we've got the list & email, so now to process the action
      if ($action == 'subscribe' || $action == 'unsubscribe') { 
        if ($action=='subscribe')   $this->subscribeUnsubscribe($trans, false);
        if ($action=='unsubscribe') $this->subscribeUnsubscribe($trans, true);
        trigger_error('--- end ---');
        exit;
      }

      // I have absolutely no idea what the user is
      // trying to do.  Give up and display an error.
      $error = 'No idea what you want me to do. Sorry!';
      $this->show($this->templateError, $trans, $error);
      trigger_error('--- end ---');
      exit;
    }

    $this->displayGenericPage($trans);
    trigger_error('--- end ---');
    exit;     
  }
}



$object = new WebScriber('config.txt');
$object->run();

?>
